"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Use an Amazon SQS queue as an event source for AWS Lambda.
 */
class SqsEventSource {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 10)) {
            throw new Error(`Maximum batch size must be between 1 and 10 inclusive (given ${this.props.batchSize})`);
        }
    }
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`SqsEventSource:${this.queue.node.uniqueId}`, {
            batchSize: this.props.batchSize,
            eventSourceArn: this.queue.queueArn,
        });
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.queue.grantConsumeMessages(target);
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('SqsEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.SqsEventSource = SqsEventSource;
//# sourceMappingURL=data:application/json;base64,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