"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const MAX_POLICY_NAME_LEN = 128;
function undefinedIfEmpty(f) {
    return core_1.Lazy.listValue({ produce: () => {
            const array = f();
            return (array && array.length > 0) ? array : undefined;
        } });
}
exports.undefinedIfEmpty = undefinedIfEmpty;
/**
 * Used to generate a unique policy name based on the policy resource construct.
 * The logical ID of the resource is a great candidate as long as it doesn't exceed
 * 128 characters, so we take the last 128 characters (in order to make sure the hash
 * is there).
 */
function generatePolicyName(scope, logicalId) {
    // as logicalId is itself a Token, resolve it first
    const resolvedLogicalId = core_1.Tokenization.resolve(logicalId, {
        scope,
        resolver: new core_1.DefaultTokenResolver(new core_1.StringConcat()),
    });
    return lastNCharacters(resolvedLogicalId, MAX_POLICY_NAME_LEN);
}
exports.generatePolicyName = generatePolicyName;
/**
 * Returns a string composed of the last n characters of str.
 * If str is shorter than n, returns str.
 *
 * @param str the string to return the last n characters of
 * @param n how many characters to return
 */
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex, str.length);
}
/**
 * Helper class that maintains the set of attached policies for a principal.
 */
class AttachedPolicies {
    constructor() {
        this.policies = new Array();
    }
    /**
     * Adds a policy to the list of attached policies.
     *
     * If this policy is already, attached, returns false.
     * If there is another policy attached with the same name, throws an exception.
     */
    attach(policy) {
        if (this.policies.find(p => p === policy)) {
            return; // already attached
        }
        if (this.policies.find(p => p.policyName === policy.policyName)) {
            throw new Error(`A policy named "${policy.policyName}" is already attached`);
        }
        this.policies.push(policy);
    }
}
exports.AttachedPolicies = AttachedPolicies;
/**
 * Merge two dictionaries that represent IAM principals
 */
function mergePrincipal(target, source) {
    for (const key of Object.keys(source)) {
        target[key] = target[key] || [];
        let value = source[key];
        if (!Array.isArray(value)) {
            value = [value];
        }
        target[key].push(...value);
    }
    return target;
}
exports.mergePrincipal = mergePrincipal;
//# sourceMappingURL=data:application/json;base64,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