"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const principals_1 = require("./principals");
const util_1 = require("./util");
/**
 * Define a new IAM user
 */
class User extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.userName,
        });
        this.grantPrincipal = this;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.groups = new Array();
        this.managedPolicies = new Array();
        this.attachedPolicies = new util_1.AttachedPolicies();
        this.managedPolicies.push(...props.managedPolicies || []);
        this.permissionsBoundary = props.permissionsBoundary;
        const user = new iam_generated_1.CfnUser(this, 'Resource', {
            userName: this.physicalName,
            groups: util_1.undefinedIfEmpty(() => this.groups),
            managedPolicyArns: core_1.Lazy.listValue({ produce: () => this.managedPolicies.map(p => p.managedPolicyArn) }, { omitEmpty: true }),
            path: props.path,
            permissionsBoundary: this.permissionsBoundary ? this.permissionsBoundary.managedPolicyArn : undefined,
            loginProfile: this.parseLoginProfile(props),
        });
        this.userName = this.getResourceNameAttribute(user.ref);
        this.userArn = this.getResourceArnAttribute(user.attrArn, {
            region: '',
            service: 'iam',
            resource: 'user',
            resourceName: this.physicalName,
        });
        this.policyFragment = new principals_1.ArnPrincipal(this.userArn).policyFragment;
        if (props.groups) {
            props.groups.forEach(g => this.addToGroup(g));
        }
    }
    /**
     * Import an existing user given a username.
     *
     * @param scope construct scope
     * @param id construct id
     * @param userName the username of the existing user to import
     */
    static fromUserName(scope, id, userName) {
        const arn = core_1.Stack.of(scope).formatArn({
            service: 'iam',
            region: '',
            resource: 'user',
            resourceName: userName,
        });
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.grantPrincipal = this;
                this.userName = userName;
                this.userArn = arn;
                this.assumeRoleAction = 'sts:AssumeRole';
                this.policyFragment = new principals_1.ArnPrincipal(arn).policyFragment;
            }
            addToPolicy(statement) {
                return this.addToPrincipalPolicy(statement).statementAdded;
            }
            addToPrincipalPolicy(statement) {
                if (!this.defaultPolicy) {
                    this.defaultPolicy = new policy_1.Policy(this, 'Policy');
                    this.defaultPolicy.attachToUser(this);
                }
                this.defaultPolicy.addStatements(statement);
                return { statementAdded: true, policyDependable: this.defaultPolicy };
            }
            addToGroup(_group) {
                throw new Error('Cannot add imported User to Group');
            }
            attachInlinePolicy(_policy) {
                throw new Error('Cannot add inline policy to imported User');
            }
            addManagedPolicy(_policy) {
                throw new Error('Cannot add managed policy to imported User');
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds this user to a group.
     */
    addToGroup(group) {
        this.groups.push(group.groupName);
    }
    /**
     * Attaches a managed policy to the user.
     * @param policy The managed policy to attach.
     */
    addManagedPolicy(policy) {
        if (this.managedPolicies.find(mp => mp === policy)) {
            return;
        }
        this.managedPolicies.push(policy);
    }
    /**
     * Attaches a policy to this user.
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToUser(this);
    }
    /**
     * Adds an IAM statement to the default policy.
     *
     * @returns true
     */
    addToPrincipalPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.defaultPolicy.attachToUser(this);
        }
        this.defaultPolicy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.defaultPolicy };
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    parseLoginProfile(props) {
        if (props.password) {
            return {
                password: props.password.toString(),
                passwordResetRequired: props.passwordResetRequired,
            };
        }
        if (props.passwordResetRequired) {
            throw new Error('Cannot set "passwordResetRequired" without specifying "initialPassword"');
        }
        return undefined; // no console access
    }
}
exports.User = User;
//# sourceMappingURL=data:application/json;base64,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