"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const diff_1 = require("./diff");
const external_1 = require("./external");
async function handler(event) {
    if (event.RequestType === 'Create') {
        return onCreate(event);
    }
    if (event.RequestType === 'Update') {
        return onUpdate(event);
    }
    if (event.RequestType === 'Delete') {
        return onDelete(event);
    }
    throw new Error('invalid request type');
}
exports.handler = handler;
async function onCreate(event) {
    var _a, _b;
    const issuerUrl = event.ResourceProperties.Url;
    const thumbprints = ((_a = event.ResourceProperties.ThumbprintList) !== null && _a !== void 0 ? _a : []).sort(); // keep sorted for UPDATE
    const clients = ((_b = event.ResourceProperties.ClientIDList) !== null && _b !== void 0 ? _b : []).sort();
    if (thumbprints.length === 0) {
        thumbprints.push(await external_1.external.downloadThumbprint(issuerUrl));
    }
    const resp = await external_1.external.createOpenIDConnectProvider({
        Url: issuerUrl,
        ClientIDList: clients,
        ThumbprintList: thumbprints,
    });
    return {
        PhysicalResourceId: resp.OpenIDConnectProviderArn,
    };
}
async function onUpdate(event) {
    var _a, _b;
    const issuerUrl = event.ResourceProperties.Url;
    const thumbprints = ((_a = event.ResourceProperties.ThumbprintList) !== null && _a !== void 0 ? _a : []).sort(); // keep sorted for UPDATE
    const clients = ((_b = event.ResourceProperties.ClientIDList) !== null && _b !== void 0 ? _b : []).sort();
    // determine which update we are talking about.
    const oldIssuerUrl = event.OldResourceProperties.Url;
    // if this is a URL update, then we basically create a new resource and cfn will delete the old one
    // since the physical resource ID will change.
    if (oldIssuerUrl !== issuerUrl) {
        return onCreate({ ...event, RequestType: 'Create' });
    }
    const providerArn = event.PhysicalResourceId;
    // if thumbprints changed, we can update in-place, but bear in mind that if the new thumbprint list
    // is empty, we will grab it from the server like we do in CREATE
    const oldThumbprints = (event.OldResourceProperties.ThumbprintList || []).sort();
    if (JSON.stringify(oldThumbprints) !== JSON.stringify(thumbprints)) {
        const thumbprintList = thumbprints.length > 0 ? thumbprints : [await external_1.external.downloadThumbprint(issuerUrl)];
        external_1.external.log('updating thumbprint list from', oldThumbprints, 'to', thumbprints);
        await external_1.external.updateOpenIDConnectProviderThumbprint({
            OpenIDConnectProviderArn: providerArn,
            ThumbprintList: thumbprintList,
        });
        // don't return, we might have more updates...
    }
    // if client ID list has changed, determine "diff" because the API is add/remove
    const oldClients = (event.OldResourceProperties.ClientIDList || []).sort();
    const diff = diff_1.arrayDiff(oldClients, clients);
    external_1.external.log(`client ID diff: ${JSON.stringify(diff)}`);
    for (const addClient of diff.adds) {
        external_1.external.log(`adding client id "${addClient}" to provider ${providerArn}`);
        await external_1.external.addClientIDToOpenIDConnectProvider({
            OpenIDConnectProviderArn: providerArn,
            ClientID: addClient,
        });
    }
    for (const deleteClient of diff.deletes) {
        external_1.external.log(`removing client id "${deleteClient}" from provider ${providerArn}`);
        await external_1.external.removeClientIDFromOpenIDConnectProvider({
            OpenIDConnectProviderArn: providerArn,
            ClientID: deleteClient,
        });
    }
    return;
}
async function onDelete(deleteEvent) {
    await external_1.external.deleteOpenIDConnectProvider({
        OpenIDConnectProviderArn: deleteEvent.PhysicalResourceId,
    });
}
//# sourceMappingURL=data:application/json;base64,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