"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const iam_generated_1 = require("./iam.generated");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * Managed policy
 *
 */
class ManagedPolicy extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.managedPolicyName,
        });
        /**
         * The policy document.
         */
        this.document = new policy_document_1.PolicyDocument();
        this.roles = new Array();
        this.users = new Array();
        this.groups = new Array();
        this.description = props.description || '';
        this.path = props.path || '/';
        if (props.document) {
            this.document = props.document;
        }
        const resource = new iam_generated_1.CfnManagedPolicy(this, 'Resource', {
            policyDocument: this.document,
            managedPolicyName: this.physicalName,
            description: this.description,
            path: this.path,
            roles: util_1.undefinedIfEmpty(() => this.roles.map(r => r.roleName)),
            users: util_1.undefinedIfEmpty(() => this.users.map(u => u.userName)),
            groups: util_1.undefinedIfEmpty(() => this.groups.map(g => g.groupName)),
        });
        if (props.users) {
            props.users.forEach(u => this.attachToUser(u));
        }
        if (props.groups) {
            props.groups.forEach(g => this.attachToGroup(g));
        }
        if (props.roles) {
            props.roles.forEach(r => this.attachToRole(r));
        }
        if (props.statements) {
            props.statements.forEach(p => this.addStatements(p));
        }
        // arn:aws:iam::123456789012:policy/teststack-CreateTestDBPolicy-16M23YE3CS700
        this.managedPolicyName = this.getResourceNameAttribute(core_1.Stack.of(this).parseArn(resource.ref, '/').resourceName);
        this.managedPolicyArn = this.getResourceArnAttribute(resource.ref, {
            region: '',
            service: 'iam',
            resource: 'policy',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a customer managed policy from the managedPolicyName.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     */
    static fromManagedPolicyName(scope, id, managedPolicyName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = core_1.Stack.of(scope).formatArn({
                    service: 'iam',
                    region: '',
                    account: core_1.Stack.of(scope).account,
                    resource: 'policy',
                    resourceName: managedPolicyName,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an external managed policy by ARN.
     *
     * For this managed policy, you only need to know the ARN to be able to use it.
     * This can be useful if you got the ARN from a CloudFormation Export.
     *
     * If the imported Managed Policy ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * managed policy has a `path` (like `arn:...:policy/AdminPolicy/AdminAllow`), the
     * `managedPolicyName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Managed Policy ARN should be supplied without the
     * `path` in order to resolve the correct managed policy resource.
     *
     * @param scope construct scope
     * @param id construct id
     * @param managedPolicyArn the ARN of the managed policy to import
     */
    static fromManagedPolicyArn(scope, id, managedPolicyArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = managedPolicyArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a managed policy from one of the policies that AWS manages.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     * Some managed policy names start with "service-role/", some start with
     * "job-function/", and some don't start with anything. Do include the
     * prefix when constructing this object.
     */
    static fromAwsManagedPolicyName(managedPolicyName) {
        class AwsManagedPolicy {
            constructor() {
                this.managedPolicyArn = core_1.Lazy.stringValue({
                    produce(ctx) {
                        return core_1.Stack.of(ctx.scope).formatArn({
                            service: 'iam',
                            region: '',
                            account: 'aws',
                            resource: 'policy',
                            resourceName: managedPolicyName,
                        });
                    },
                });
            }
        }
        return new AwsManagedPolicy();
    }
    /**
     * Adds a statement to the policy document.
     */
    addStatements(...statement) {
        this.document.addStatements(...statement);
    }
    /**
     * Attaches this policy to a user.
     */
    attachToUser(user) {
        if (this.users.find(u => u === user)) {
            return;
        }
        this.users.push(user);
    }
    /**
     * Attaches this policy to a role.
     */
    attachToRole(role) {
        if (this.roles.find(r => r === role)) {
            return;
        }
        this.roles.push(role);
    }
    /**
     * Attaches this policy to a group.
     */
    attachToGroup(group) {
        if (this.groups.find(g => g === group)) {
            return;
        }
        this.groups.push(group);
    }
    validate() {
        const result = new Array();
        // validate that the policy document is not empty
        if (this.document.isEmpty) {
            result.push('Managed Policy is empty. You must add statements to the policy');
        }
        return result;
    }
}
exports.ManagedPolicy = ManagedPolicy;
//# sourceMappingURL=data:application/json;base64,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