"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const file_system_1 = require("./file-system");
const fsx_generated_1 = require("./fsx.generated");
/**
 * The different kinds of file system deployments used by Lustre.
 */
var LustreDeploymentType;
(function (LustreDeploymentType) {
    /**
     * Original type for shorter term data processing. Data is not replicated and does not persist on server fail.
     */
    LustreDeploymentType["SCRATCH_1"] = "SCRATCH_1";
    /**
     * Newer type for shorter term data processing. Data is not replicated and does not persist on server fail.
     * Provides better support for spiky workloads.
     */
    LustreDeploymentType["SCRATCH_2"] = "SCRATCH_2";
    /**
     * Long term storage. Data is replicated and file servers are replaced if they fail.
     */
    LustreDeploymentType["PERSISTENT_1"] = "PERSISTENT_1";
})(LustreDeploymentType = exports.LustreDeploymentType || (exports.LustreDeploymentType = {}));
/**
 * The FSx for Lustre File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 *
 * @resource AWS::FSx::FileSystem
 */
class LustreFileSystem extends file_system_1.FileSystemBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.validateProps(props);
        const updatedLustureProps = {
            importedFileChunkSize: props.lustreConfiguration.importedFileChunkSizeMiB,
            weeklyMaintenanceStartTime: (_a = props.lustreConfiguration.weeklyMaintenanceStartTime) === null || _a === void 0 ? void 0 : _a.toTimestamp(),
        };
        const lustreConfiguration = Object.assign({}, props.lustreConfiguration, updatedLustureProps);
        const securityGroup = (props.securityGroup || new aws_ec2_1.SecurityGroup(this, 'FsxLustreSecurityGroup', {
            vpc: props.vpc,
        }));
        securityGroup.addIngressRule(securityGroup, aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort));
        this.connections = LustreFileSystem.configureConnections(securityGroup);
        this.fileSystem = new fsx_generated_1.CfnFileSystem(this, 'Resource', {
            fileSystemType: LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: [props.vpcSubnet.subnetId],
            backupId: props.backupId,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lustreConfiguration,
            securityGroupIds: [securityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        this.fileSystemId = this.fileSystem.ref;
        this.dnsName = `${this.fileSystemId}.fsx.${this.stack.region}.${core_1.Aws.URL_SUFFIX}`;
        this.mountName = this.fileSystem.attrLustreMountName;
    }
    /**
     * Import an existing FSx for Lustre file system from the given properties.
     */
    static fromLustreFileSystemAttributes(scope, id, attrs) {
        class Import extends file_system_1.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = LustreFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object with all the ports required by FSx for Lustre
     */
    static configureConnections(securityGroup) {
        const connections = new aws_ec2_1.Connections({
            securityGroups: [securityGroup],
            defaultPort: aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort),
        });
        return connections;
    }
    /**
     * Validates the props provided for a new FSx for Lustre file system.
     */
    validateProps(props) {
        const lustreConfiguration = props.lustreConfiguration;
        const deploymentType = lustreConfiguration.deploymentType;
        // Make sure the import path is valid before validating the export path
        this.validateImportPath(lustreConfiguration.importPath);
        this.validateExportPath(lustreConfiguration.exportPath, lustreConfiguration.importPath);
        this.validateImportedFileChunkSize(lustreConfiguration.importedFileChunkSizeMiB);
        this.validatePerUnitStorageThroughput(deploymentType, lustreConfiguration.perUnitStorageThroughput);
        this.validateStorageCapacity(deploymentType, props.storageCapacityGiB);
    }
    /**
     * Validates the export path is in the correct format and matches the import path.
     */
    validateExportPath(exportPath, importPath) {
        if (exportPath === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('Cannot define an export path without also defining an import path');
        }
        if (core_1.Token.isUnresolved(exportPath) && core_1.Token.isUnresolved(importPath)) {
            return;
        }
        if (core_1.Token.isUnresolved(importPath) !== core_1.Token.isUnresolved(exportPath)) {
            throw new Error('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
        }
        if (!exportPath.startsWith(importPath)) {
            throw new Error(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
        }
        if (exportPath.length > 900) {
            throw new Error(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the importedFileChunkSize is in the correct range.
     */
    validateImportedFileChunkSize(importedFileChunkSize) {
        if (importedFileChunkSize === undefined) {
            return;
        }
        if (importedFileChunkSize < 1 || importedFileChunkSize > 512000) {
            throw new Error(`importedFileChunkSize cannot be ${importedFileChunkSize} MiB. It must be a value from 1 to 512,000 MiB`);
        }
    }
    /**
     * Validates the import path is the correct format.
     */
    validateImportPath(importPath) {
        if (importPath === undefined || core_1.Token.isUnresolved(importPath)) {
            return;
        }
        const regexp = /^s3:\/\//;
        if (importPath.search(regexp) === -1) {
            throw new Error(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
        }
        if (importPath.length > 900) {
            throw new Error(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the perUnitStorageThroughput is defined correctly for the given deploymentType.
     */
    validatePerUnitStorageThroughput(deploymentType, perUnitStorageThroughput) {
        if (perUnitStorageThroughput === undefined) {
            return;
        }
        if (deploymentType !== LustreDeploymentType.PERSISTENT_1) {
            throw new Error('perUnitStorageThroughput can only be set for the PERSISTENT_1 deployment type');
        }
        if (![50, 100, 200].includes(perUnitStorageThroughput)) {
            throw new Error('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB');
        }
    }
    /**
     * Validates the storage capacity is an acceptable value for the deployment type.
     */
    validateStorageCapacity(deploymentType, storageCapacity) {
        if (deploymentType === LustreDeploymentType.SCRATCH_1) {
            if (![1200, 2400, 3600].includes(storageCapacity) && storageCapacity % 3600 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600');
            }
        }
        else {
            if (![1200, 2400].includes(storageCapacity) && storageCapacity % 2400 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, or a multiple of 2,400');
            }
        }
    }
}
exports.LustreFileSystem = LustreFileSystem;
/**
 * The default FSx file system type used by FSx for Lustre.
 */
LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'LUSTRE';
/**
 * The default ports the file system listens on. Actual port list is: [988, 1021, 1022, 1023]
 */
LustreFileSystem.DEFAULT_PORT_RANGE = { startPort: 988, endPort: 1023 };
//# sourceMappingURL=data:application/json;base64,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