"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const ecs = require("../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const util_1 = require("./util");
/**
 * Start a task on an ECS cluster
 */
class EcsTask {
    constructor(props) {
        this.props = props;
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        this.cluster = props.cluster;
        this.taskDefinition = props.taskDefinition;
        this.taskCount = props.taskCount !== undefined ? props.taskCount : 1;
        // Security groups are only configurable with the "awsvpc" network mode.
        if (this.taskDefinition.networkMode !== ecs.NetworkMode.AWS_VPC) {
            if (props.securityGroup !== undefined || props.securityGroups !== undefined) {
                this.taskDefinition.node.addWarning('security groups are ignored when network mode is not awsvpc');
            }
            return;
        }
        if (props.securityGroups) {
            this.securityGroups = props.securityGroups;
            return;
        }
        let securityGroup = props.securityGroup || this.taskDefinition.node.tryFindChild('SecurityGroup');
        securityGroup = securityGroup || new ec2.SecurityGroup(this.taskDefinition, 'SecurityGroup', { vpc: this.props.cluster.vpc });
        this.securityGroup = securityGroup; // Maintain backwards-compatibility for customers that read the generated security group.
        this.securityGroups = [securityGroup];
    }
    /**
     * Allows using tasks as target of CloudWatch events
     */
    bind(_rule, _id) {
        const policyStatements = [new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.taskDefinition.taskDefinitionArn],
                conditions: {
                    ArnEquals: { 'ecs:cluster': this.cluster.clusterArn },
                },
            })];
        // If it so happens that a Task Execution Role was created for the TaskDefinition,
        // then the CloudWatch Events Role must have permissions to pass it (otherwise it doesn't).
        if (this.taskDefinition.executionRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.executionRole.roleArn],
            }));
        }
        // For Fargate task we need permission to pass the task role.
        if (this.taskDefinition.isFargateCompatible) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.taskRole.roleArn],
            }));
        }
        const arn = this.cluster.clusterArn;
        const role = util_1.singletonEventRole(this.taskDefinition, policyStatements);
        const containerOverrides = this.props.containerOverrides && this.props.containerOverrides
            .map(({ containerName, ...overrides }) => ({ name: containerName, ...overrides }));
        const input = { containerOverrides };
        const taskCount = this.taskCount;
        const taskDefinitionArn = this.taskDefinition.taskDefinitionArn;
        const subnetSelection = this.props.subnetSelection || { subnetType: ec2.SubnetType.PRIVATE };
        const assignPublicIp = subnetSelection.subnetType === ec2.SubnetType.PUBLIC ? 'ENABLED' : 'DISABLED';
        const baseEcsParameters = { taskCount, taskDefinitionArn };
        const ecsParameters = this.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC
            ? {
                ...baseEcsParameters,
                launchType: this.taskDefinition.isEc2Compatible ? 'EC2' : 'FARGATE',
                networkConfiguration: {
                    awsVpcConfiguration: {
                        subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                        assignPublicIp,
                        securityGroups: this.securityGroups && this.securityGroups.map(sg => sg.securityGroupId),
                    },
                },
            }
            : baseEcsParameters;
        return {
            id: '',
            arn,
            role,
            ecsParameters,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: this.taskDefinition,
        };
    }
}
exports.EcsTask = EcsTask;
//# sourceMappingURL=data:application/json;base64,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