"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const aws_iam_1 = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_load_balancer_1 = require("../shared/base-load-balancer");
const network_listener_1 = require("./network-listener");
/**
 * Define a new network load balancer
 *
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class NetworkLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    static fromNetworkLoadBalancerAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.loadBalancerArn = attrs.loadBalancerArn;
                this.vpc = attrs.vpc;
            }
            addListener(lid, props) {
                return new network_listener_1.NetworkListener(this, lid, {
                    loadBalancer: this,
                    ...props,
                });
            }
            get loadBalancerCanonicalHostedZoneId() {
                if (attrs.loadBalancerCanonicalHostedZoneId) {
                    return attrs.loadBalancerCanonicalHostedZoneId;
                }
                // tslint:disable-next-line:max-line-length
                throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
            get loadBalancerDnsName() {
                if (attrs.loadBalancerDnsName) {
                    return attrs.loadBalancerDnsName;
                }
                // tslint:disable-next-line:max-line-length
                throw new Error(`'loadBalancerDnsName' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'network',
        });
        if (props.crossZoneEnabled) {
            this.setAttribute('load_balancing.cross_zone.enabled', 'true');
        }
    }
    /**
     * Add a listener to this load balancer
     *
     * @returns The newly created listener
     */
    addListener(id, props) {
        return new network_listener_1.NetworkListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     *
     * This is extending the BaseLoadBalancer.logAccessLogs method to match the bucket permissions described
     * at https://docs.aws.amazon.com/elasticloadbalancing/latest/network/load-balancer-access-logs.html#access-logging-bucket-requirements
     */
    logAccessLogs(bucket, prefix) {
        super.logAccessLogs(bucket, prefix);
        const logsDeliveryServicePrincipal = new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com');
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [logsDeliveryServicePrincipal],
            resources: [
                bucket.arnForObjects(`${prefix ? prefix + '/' : ''}AWSLogs/${this.stack.account}/*`),
            ],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:GetBucketAcl'],
            principals: [logsDeliveryServicePrincipal],
            resources: [bucket.bucketArn],
        }));
    }
    /**
     * Return the given named metric for this Network Load Balancer
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props,
        }).attachTo(this);
    }
    /**
     * The total number of concurrent TCP flows (or connections) from clients to targets.
     *
     * This metric includes connections in the SYN_SENT and ESTABLISHED states.
     * TCP connections are not terminated at the load balancer, so a client
     * opening a TCP connection to a target counts as a single flow.
     *
     * @default Average over 5 minutes
     */
    metricActiveFlowCount(props) {
        return this.metric('ActiveFlowCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricConsumedLCUs(props) {
        return this.metric('ConsumedLCUs', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     */
    metricUnHealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The total number of new TCP flows (or connections) established from clients to targets in the time period.
     *
     * @default Sum over 5 minutes
     */
    metricNewFlowCount(props) {
        return this.metric('NewFlowCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of bytes processed by the load balancer, including TCP/IP headers.
     *
     * @default Sum over 5 minutes
     */
    metricProcessedBytes(props) {
        return this.metric('ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of reset (RST) packets sent from a client to a target.
     *
     * These resets are generated by the client and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricTcpClientResetCount(props) {
        return this.metric('TCP_Client_Reset_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of reset (RST) packets generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricTcpElbResetCount(props) {
        return this.metric('TCP_ELB_Reset_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of reset (RST) packets sent from a target to a client.
     *
     * These resets are generated by the target and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricTcpTargetResetCount(props) {
        return this.metric('TCP_Target_Reset_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
}
exports.NetworkLoadBalancer = NetworkLoadBalancer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmV0d29yay1sb2FkLWJhbGFuY2VyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibmV0d29yay1sb2FkLWJhbGFuY2VyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsc0RBQXNELENBQUMsMERBQTBEO0FBRWpILDhDQUFxRSxDQUFDLG1EQUFtRDtBQUV6SCx3Q0FBb0QsQ0FBQyxnREFBZ0Q7QUFDckcscUVBQXdHO0FBQ3hHLHlEQUErRTtBQXdDL0U7Ozs7R0FJRztBQUNILE1BQWEsbUJBQW9CLFNBQVEscUNBQWdCO0lBQzlDLE1BQU0sQ0FBQyxpQ0FBaUMsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFvQztRQUM5RyxNQUFNLE1BQU8sU0FBUSxlQUFRO1lBQTdCOztnQkFDb0Isb0JBQWUsR0FBRyxLQUFLLENBQUMsZUFBZSxDQUFDO2dCQUN4QyxRQUFHLEdBQWMsS0FBSyxDQUFDLEdBQUcsQ0FBQztZQXFCL0MsQ0FBQztZQXBCVSxXQUFXLENBQUMsR0FBVyxFQUFFLEtBQStCO2dCQUMzRCxPQUFPLElBQUksa0NBQWUsQ0FBQyxJQUFJLEVBQUUsR0FBRyxFQUFFO29CQUNsQyxZQUFZLEVBQUUsSUFBSTtvQkFDbEIsR0FBRyxLQUFLO2lCQUNYLENBQUMsQ0FBQztZQUNQLENBQUM7WUFDRCxJQUFXLGlDQUFpQztnQkFDeEMsSUFBSSxLQUFLLENBQUMsaUNBQWlDLEVBQUU7b0JBQ3pDLE9BQU8sS0FBSyxDQUFDLGlDQUFpQyxDQUFDO2lCQUNsRDtnQkFDRCwyQ0FBMkM7Z0JBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0dBQWdHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxrQkFBa0IsQ0FBQyxDQUFDO1lBQ3RKLENBQUM7WUFDRCxJQUFXLG1CQUFtQjtnQkFDMUIsSUFBSSxLQUFLLENBQUMsbUJBQW1CLEVBQUU7b0JBQzNCLE9BQU8sS0FBSyxDQUFDLG1CQUFtQixDQUFDO2lCQUNwQztnQkFDRCwyQ0FBMkM7Z0JBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsa0ZBQWtGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxrQkFBa0IsQ0FBQyxDQUFDO1lBQ3hJLENBQUM7U0FDSjtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFDRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQStCO1FBQ3JFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRTtZQUNwQixJQUFJLEVBQUUsU0FBUztTQUNsQixDQUFDLENBQUM7UUFDSCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUN4QixJQUFJLENBQUMsWUFBWSxDQUFDLG1DQUFtQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1NBQ2xFO0lBQ0wsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsRUFBVSxFQUFFLEtBQStCO1FBQzFELE9BQU8sSUFBSSxrQ0FBZSxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDakMsWUFBWSxFQUFFLElBQUk7WUFDbEIsR0FBRyxLQUFLO1NBQ1gsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7Ozs7OztPQVFHO0lBQ0ksYUFBYSxDQUFDLE1BQWUsRUFBRSxNQUFlO1FBQ2pELEtBQUssQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3BDLE1BQU0sNEJBQTRCLEdBQUcsSUFBSSwwQkFBZ0IsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3pGLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLHlCQUFlLENBQUM7WUFDM0MsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO1lBQ3pCLFVBQVUsRUFBRSxDQUFDLDRCQUE0QixDQUFDO1lBQzFDLFNBQVMsRUFBRTtnQkFDUCxNQUFNLENBQUMsYUFBYSxDQUFDLEdBQUcsTUFBTSxDQUFDLENBQUMsQ0FBQyxNQUFNLEdBQUcsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLFdBQVcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksQ0FBQzthQUN2RjtZQUNELFVBQVUsRUFBRTtnQkFDUixZQUFZLEVBQUUsRUFBRSxjQUFjLEVBQUUsMkJBQTJCLEVBQUU7YUFDaEU7U0FDSixDQUFDLENBQUMsQ0FBQztRQUNKLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLHlCQUFlLENBQUM7WUFDM0MsT0FBTyxFQUFFLENBQUMsaUJBQWlCLENBQUM7WUFDNUIsVUFBVSxFQUFFLENBQUMsNEJBQTRCLENBQUM7WUFDMUMsU0FBUyxFQUFFLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQztTQUNoQyxDQUFDLENBQUMsQ0FBQztJQUNSLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFVBQWtCLEVBQUUsS0FBZ0M7UUFDOUQsT0FBTyxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUM7WUFDekIsU0FBUyxFQUFFLGdCQUFnQjtZQUMzQixVQUFVO1lBQ1YsVUFBVSxFQUFFLEVBQUUsWUFBWSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsRUFBRTtZQUN2RCxHQUFHLEtBQUs7U0FDWCxDQUFDLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFDRDs7Ozs7Ozs7T0FRRztJQUNJLHFCQUFxQixDQUFDLEtBQWdDO1FBQ3pELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRTtZQUNsQyxTQUFTLEVBQUUsU0FBUztZQUNwQixHQUFHLEtBQUs7U0FDWCxDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLGtCQUFrQixDQUFDLEtBQWdDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxjQUFjLEVBQUU7WUFDL0IsU0FBUyxFQUFFLEtBQUs7WUFDaEIsR0FBRyxLQUFLO1NBQ1gsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxzQkFBc0IsQ0FBQyxLQUFnQztRQUMxRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsa0JBQWtCLEVBQUU7WUFDbkMsU0FBUyxFQUFFLFNBQVM7WUFDcEIsR0FBRyxLQUFLO1NBQ1gsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSx3QkFBd0IsQ0FBQyxLQUFnQztRQUM1RCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsb0JBQW9CLEVBQUU7WUFDckMsU0FBUyxFQUFFLFNBQVM7WUFDcEIsR0FBRyxLQUFLO1NBQ1gsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxrQkFBa0IsQ0FBQyxLQUFnQztRQUN0RCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsY0FBYyxFQUFFO1lBQy9CLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLEdBQUcsS0FBSztTQUNYLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksb0JBQW9CLENBQUMsS0FBZ0M7UUFDeEQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGdCQUFnQixFQUFFO1lBQ2pDLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLEdBQUcsS0FBSztTQUNYLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUIsQ0FBQyxLQUFnQztRQUM3RCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsd0JBQXdCLEVBQUU7WUFDekMsU0FBUyxFQUFFLEtBQUs7WUFDaEIsR0FBRyxLQUFLO1NBQ1gsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxzQkFBc0IsQ0FBQyxLQUFnQztRQUMxRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMscUJBQXFCLEVBQUU7WUFDdEMsU0FBUyxFQUFFLEtBQUs7WUFDaEIsR0FBRyxLQUFLO1NBQ1gsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QixDQUFDLEtBQWdDO1FBQzdELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyx3QkFBd0IsRUFBRTtZQUN6QyxTQUFTLEVBQUUsS0FBSztZQUNoQixHQUFHLEtBQUs7U0FDWCxDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUFuTUQsa0RBbU1DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2xvdWR3YXRjaCBmcm9tIFwiLi4vLi4vLi4vYXdzLWNsb3Vkd2F0Y2hcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJ1xuaW1wb3J0ICogYXMgZWMyIGZyb20gXCIuLi8uLi8uLi9hd3MtZWMyXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJ1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBTZXJ2aWNlUHJpbmNpcGFsIH0gZnJvbSBcIi4uLy4uLy4uL2F3cy1pYW1cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nXG5pbXBvcnQgeyBJQnVja2V0IH0gZnJvbSBcIi4uLy4uLy4uL2F3cy1zM1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJ1xuaW1wb3J0IHsgQ29uc3RydWN0LCBSZXNvdXJjZSB9IGZyb20gXCIuLi8uLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQmFzZUxvYWRCYWxhbmNlciwgQmFzZUxvYWRCYWxhbmNlclByb3BzLCBJTG9hZEJhbGFuY2VyVjIgfSBmcm9tICcuLi9zaGFyZWQvYmFzZS1sb2FkLWJhbGFuY2VyJztcbmltcG9ydCB7IEJhc2VOZXR3b3JrTGlzdGVuZXJQcm9wcywgTmV0d29ya0xpc3RlbmVyIH0gZnJvbSAnLi9uZXR3b3JrLWxpc3RlbmVyJztcbi8qKlxuICogUHJvcGVydGllcyBmb3IgYSBuZXR3b3JrIGxvYWQgYmFsYW5jZXJcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOZXR3b3JrTG9hZEJhbGFuY2VyUHJvcHMgZXh0ZW5kcyBCYXNlTG9hZEJhbGFuY2VyUHJvcHMge1xuICAgIC8qKlxuICAgICAqIEluZGljYXRlcyB3aGV0aGVyIGNyb3NzLXpvbmUgbG9hZCBiYWxhbmNpbmcgaXMgZW5hYmxlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgY3Jvc3Nab25lRW5hYmxlZD86IGJvb2xlYW47XG59XG4vKipcbiAqIFByb3BlcnRpZXMgdG8gcmVmZXJlbmNlIGFuIGV4aXN0aW5nIGxvYWQgYmFsYW5jZXJcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOZXR3b3JrTG9hZEJhbGFuY2VyQXR0cmlidXRlcyB7XG4gICAgLyoqXG4gICAgICogQVJOIG9mIHRoZSBsb2FkIGJhbGFuY2VyXG4gICAgICovXG4gICAgcmVhZG9ubHkgbG9hZEJhbGFuY2VyQXJuOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGNhbm9uaWNhbCBob3N0ZWQgem9uZSBJRCBvZiB0aGlzIGxvYWQgYmFsYW5jZXJcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gV2hlbiBub3QgcHJvdmlkZWQsIExCIGNhbm5vdCBiZSB1c2VkIGFzIFJvdXRlNTMgQWxpYXMgdGFyZ2V0LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGxvYWRCYWxhbmNlckNhbm9uaWNhbEhvc3RlZFpvbmVJZD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgRE5TIG5hbWUgb2YgdGhpcyBsb2FkIGJhbGFuY2VyXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFdoZW4gbm90IHByb3ZpZGVkLCBMQiBjYW5ub3QgYmUgdXNlZCBhcyBSb3V0ZTUzIEFsaWFzIHRhcmdldC5cbiAgICAgKi9cbiAgICByZWFkb25seSBsb2FkQmFsYW5jZXJEbnNOYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBWUEMgdG8gYXNzb2NpYXRlIHdpdGggdGhlIGxvYWQgYmFsYW5jZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFdoZW4gbm90IHByb3ZpZGVkLCBsaXN0ZW5lcnMgY2Fubm90IGJlIGNyZWF0ZWQgb24gaW1wb3J0ZWQgbG9hZFxuICAgICAqIGJhbGFuY2Vycy5cbiAgICAgKi9cbiAgICByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcbn1cbi8qKlxuICogRGVmaW5lIGEgbmV3IG5ldHdvcmsgbG9hZCBiYWxhbmNlclxuICpcbiAqIEByZXNvdXJjZSBBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OkxvYWRCYWxhbmNlclxuICovXG5leHBvcnQgY2xhc3MgTmV0d29ya0xvYWRCYWxhbmNlciBleHRlbmRzIEJhc2VMb2FkQmFsYW5jZXIgaW1wbGVtZW50cyBJTmV0d29ya0xvYWRCYWxhbmNlciB7XG4gICAgcHVibGljIHN0YXRpYyBmcm9tTmV0d29ya0xvYWRCYWxhbmNlckF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IE5ldHdvcmtMb2FkQmFsYW5jZXJBdHRyaWJ1dGVzKTogSU5ldHdvcmtMb2FkQmFsYW5jZXIge1xuICAgICAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElOZXR3b3JrTG9hZEJhbGFuY2VyIHtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBsb2FkQmFsYW5jZXJBcm4gPSBhdHRycy5sb2FkQmFsYW5jZXJBcm47XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgdnBjPzogZWMyLklWcGMgPSBhdHRycy52cGM7XG4gICAgICAgICAgICBwdWJsaWMgYWRkTGlzdGVuZXIobGlkOiBzdHJpbmcsIHByb3BzOiBCYXNlTmV0d29ya0xpc3RlbmVyUHJvcHMpOiBOZXR3b3JrTGlzdGVuZXIge1xuICAgICAgICAgICAgICAgIHJldHVybiBuZXcgTmV0d29ya0xpc3RlbmVyKHRoaXMsIGxpZCwge1xuICAgICAgICAgICAgICAgICAgICBsb2FkQmFsYW5jZXI6IHRoaXMsXG4gICAgICAgICAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgcHVibGljIGdldCBsb2FkQmFsYW5jZXJDYW5vbmljYWxIb3N0ZWRab25lSWQoKTogc3RyaW5nIHtcbiAgICAgICAgICAgICAgICBpZiAoYXR0cnMubG9hZEJhbGFuY2VyQ2Fub25pY2FsSG9zdGVkWm9uZUlkKSB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBhdHRycy5sb2FkQmFsYW5jZXJDYW5vbmljYWxIb3N0ZWRab25lSWQ7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYCdsb2FkQmFsYW5jZXJDYW5vbmljYWxIb3N0ZWRab25lSWQnIHdhcyBub3QgcHJvdmlkZWQgd2hlbiBjb25zdHJ1Y3RpbmcgTmV0d29yayBMb2FkIEJhbGFuY2VyICR7dGhpcy5ub2RlLnBhdGh9IGZyb20gYXR0cmlidXRlc2ApO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgcHVibGljIGdldCBsb2FkQmFsYW5jZXJEbnNOYW1lKCk6IHN0cmluZyB7XG4gICAgICAgICAgICAgICAgaWYgKGF0dHJzLmxvYWRCYWxhbmNlckRuc05hbWUpIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIGF0dHJzLmxvYWRCYWxhbmNlckRuc05hbWU7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYCdsb2FkQmFsYW5jZXJEbnNOYW1lJyB3YXMgbm90IHByb3ZpZGVkIHdoZW4gY29uc3RydWN0aW5nIE5ldHdvcmsgTG9hZCBCYWxhbmNlciAke3RoaXMubm9kZS5wYXRofSBmcm9tIGF0dHJpYnV0ZXNgKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICAgIH1cbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogTmV0d29ya0xvYWRCYWxhbmNlclByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMsIHtcbiAgICAgICAgICAgIHR5cGU6ICduZXR3b3JrJyxcbiAgICAgICAgfSk7XG4gICAgICAgIGlmIChwcm9wcy5jcm9zc1pvbmVFbmFibGVkKSB7XG4gICAgICAgICAgICB0aGlzLnNldEF0dHJpYnV0ZSgnbG9hZF9iYWxhbmNpbmcuY3Jvc3Nfem9uZS5lbmFibGVkJywgJ3RydWUnKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSBsaXN0ZW5lciB0byB0aGlzIGxvYWQgYmFsYW5jZXJcbiAgICAgKlxuICAgICAqIEByZXR1cm5zIFRoZSBuZXdseSBjcmVhdGVkIGxpc3RlbmVyXG4gICAgICovXG4gICAgcHVibGljIGFkZExpc3RlbmVyKGlkOiBzdHJpbmcsIHByb3BzOiBCYXNlTmV0d29ya0xpc3RlbmVyUHJvcHMpOiBOZXR3b3JrTGlzdGVuZXIge1xuICAgICAgICByZXR1cm4gbmV3IE5ldHdvcmtMaXN0ZW5lcih0aGlzLCBpZCwge1xuICAgICAgICAgICAgbG9hZEJhbGFuY2VyOiB0aGlzLFxuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBFbmFibGUgYWNjZXNzIGxvZ2dpbmcgZm9yIHRoaXMgbG9hZCBiYWxhbmNlci5cbiAgICAgKlxuICAgICAqIEEgcmVnaW9uIG11c3QgYmUgc3BlY2lmaWVkIG9uIHRoZSBzdGFjayBjb250YWluaW5nIHRoZSBsb2FkIGJhbGFuY2VyOyB5b3UgY2Fubm90IGVuYWJsZSBsb2dnaW5nIG9uXG4gICAgICogZW52aXJvbm1lbnQtYWdub3N0aWMgc3RhY2tzLiBTZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nkay9sYXRlc3QvZ3VpZGUvZW52aXJvbm1lbnRzLmh0bWxcbiAgICAgKlxuICAgICAqIFRoaXMgaXMgZXh0ZW5kaW5nIHRoZSBCYXNlTG9hZEJhbGFuY2VyLmxvZ0FjY2Vzc0xvZ3MgbWV0aG9kIHRvIG1hdGNoIHRoZSBidWNrZXQgcGVybWlzc2lvbnMgZGVzY3JpYmVkXG4gICAgICogYXQgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNsb2FkYmFsYW5jaW5nL2xhdGVzdC9uZXR3b3JrL2xvYWQtYmFsYW5jZXItYWNjZXNzLWxvZ3MuaHRtbCNhY2Nlc3MtbG9nZ2luZy1idWNrZXQtcmVxdWlyZW1lbnRzXG4gICAgICovXG4gICAgcHVibGljIGxvZ0FjY2Vzc0xvZ3MoYnVja2V0OiBJQnVja2V0LCBwcmVmaXg/OiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIubG9nQWNjZXNzTG9ncyhidWNrZXQsIHByZWZpeCk7XG4gICAgICAgIGNvbnN0IGxvZ3NEZWxpdmVyeVNlcnZpY2VQcmluY2lwYWwgPSBuZXcgU2VydmljZVByaW5jaXBhbCgnZGVsaXZlcnkubG9ncy5hbWF6b25hd3MuY29tJyk7XG4gICAgICAgIGJ1Y2tldC5hZGRUb1Jlc291cmNlUG9saWN5KG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgYWN0aW9uczogWydzMzpQdXRPYmplY3QnXSxcbiAgICAgICAgICAgIHByaW5jaXBhbHM6IFtsb2dzRGVsaXZlcnlTZXJ2aWNlUHJpbmNpcGFsXSxcbiAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgIGJ1Y2tldC5hcm5Gb3JPYmplY3RzKGAke3ByZWZpeCA/IHByZWZpeCArICcvJyA6ICcnfUFXU0xvZ3MvJHt0aGlzLnN0YWNrLmFjY291bnR9LypgKSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7ICdzMzp4LWFtei1hY2wnOiAnYnVja2V0LW93bmVyLWZ1bGwtY29udHJvbCcgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pKTtcbiAgICAgICAgYnVja2V0LmFkZFRvUmVzb3VyY2VQb2xpY3kobmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbJ3MzOkdldEJ1Y2tldEFjbCddLFxuICAgICAgICAgICAgcHJpbmNpcGFsczogW2xvZ3NEZWxpdmVyeVNlcnZpY2VQcmluY2lwYWxdLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbYnVja2V0LmJ1Y2tldEFybl0sXG4gICAgICAgIH0pKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJuIHRoZSBnaXZlbiBuYW1lZCBtZXRyaWMgZm9yIHRoaXMgTmV0d29yayBMb2FkIEJhbGFuY2VyXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBBdmVyYWdlIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgICAgICByZXR1cm4gbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgICAgICAgIG5hbWVzcGFjZTogJ0FXUy9OZXR3b3JrRUxCJyxcbiAgICAgICAgICAgIG1ldHJpY05hbWUsXG4gICAgICAgICAgICBkaW1lbnNpb25zOiB7IExvYWRCYWxhbmNlcjogdGhpcy5sb2FkQmFsYW5jZXJGdWxsTmFtZSB9LFxuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgIH0pLmF0dGFjaFRvKHRoaXMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgdG90YWwgbnVtYmVyIG9mIGNvbmN1cnJlbnQgVENQIGZsb3dzIChvciBjb25uZWN0aW9ucykgZnJvbSBjbGllbnRzIHRvIHRhcmdldHMuXG4gICAgICpcbiAgICAgKiBUaGlzIG1ldHJpYyBpbmNsdWRlcyBjb25uZWN0aW9ucyBpbiB0aGUgU1lOX1NFTlQgYW5kIEVTVEFCTElTSEVEIHN0YXRlcy5cbiAgICAgKiBUQ1AgY29ubmVjdGlvbnMgYXJlIG5vdCB0ZXJtaW5hdGVkIGF0IHRoZSBsb2FkIGJhbGFuY2VyLCBzbyBhIGNsaWVudFxuICAgICAqIG9wZW5pbmcgYSBUQ1AgY29ubmVjdGlvbiB0byBhIHRhcmdldCBjb3VudHMgYXMgYSBzaW5nbGUgZmxvdy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IEF2ZXJhZ2Ugb3ZlciA1IG1pbnV0ZXNcbiAgICAgKi9cbiAgICBwdWJsaWMgbWV0cmljQWN0aXZlRmxvd0NvdW50KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKSB7XG4gICAgICAgIHJldHVybiB0aGlzLm1ldHJpYygnQWN0aXZlRmxvd0NvdW50Jywge1xuICAgICAgICAgICAgc3RhdGlzdGljOiAnQXZlcmFnZScsXG4gICAgICAgICAgICAuLi5wcm9wcyxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFRoZSBudW1iZXIgb2YgbG9hZCBiYWxhbmNlciBjYXBhY2l0eSB1bml0cyAoTENVKSB1c2VkIGJ5IHlvdXIgbG9hZCBiYWxhbmNlci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFN1bSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWNDb25zdW1lZExDVXMocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMubWV0cmljKCdDb25zdW1lZExDVXMnLCB7XG4gICAgICAgICAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgbnVtYmVyIG9mIHRhcmdldHMgdGhhdCBhcmUgY29uc2lkZXJlZCBoZWFsdGh5LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgQXZlcmFnZSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWNIZWFsdGh5SG9zdENvdW50KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKSB7XG4gICAgICAgIHJldHVybiB0aGlzLm1ldHJpYygnSGVhbHRoeUhvc3RDb3VudCcsIHtcbiAgICAgICAgICAgIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnLFxuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgbnVtYmVyIG9mIHRhcmdldHMgdGhhdCBhcmUgY29uc2lkZXJlZCB1bmhlYWx0aHkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBBdmVyYWdlIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpY1VuSGVhbHRoeUhvc3RDb3VudChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucykge1xuICAgICAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ1VuSGVhbHRoeUhvc3RDb3VudCcsIHtcbiAgICAgICAgICAgIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnLFxuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgdG90YWwgbnVtYmVyIG9mIG5ldyBUQ1AgZmxvd3MgKG9yIGNvbm5lY3Rpb25zKSBlc3RhYmxpc2hlZCBmcm9tIGNsaWVudHMgdG8gdGFyZ2V0cyBpbiB0aGUgdGltZSBwZXJpb2QuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBTdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICAgKi9cbiAgICBwdWJsaWMgbWV0cmljTmV3Rmxvd0NvdW50KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKSB7XG4gICAgICAgIHJldHVybiB0aGlzLm1ldHJpYygnTmV3Rmxvd0NvdW50Jywge1xuICAgICAgICAgICAgc3RhdGlzdGljOiAnU3VtJyxcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogVGhlIHRvdGFsIG51bWJlciBvZiBieXRlcyBwcm9jZXNzZWQgYnkgdGhlIGxvYWQgYmFsYW5jZXIsIGluY2x1ZGluZyBUQ1AvSVAgaGVhZGVycy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFN1bSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWNQcm9jZXNzZWRCeXRlcyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucykge1xuICAgICAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ1Byb2Nlc3NlZEJ5dGVzJywge1xuICAgICAgICAgICAgc3RhdGlzdGljOiAnU3VtJyxcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogVGhlIHRvdGFsIG51bWJlciBvZiByZXNldCAoUlNUKSBwYWNrZXRzIHNlbnQgZnJvbSBhIGNsaWVudCB0byBhIHRhcmdldC5cbiAgICAgKlxuICAgICAqIFRoZXNlIHJlc2V0cyBhcmUgZ2VuZXJhdGVkIGJ5IHRoZSBjbGllbnQgYW5kIGZvcndhcmRlZCBieSB0aGUgbG9hZCBiYWxhbmNlci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFN1bSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWNUY3BDbGllbnRSZXNldENvdW50KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKSB7XG4gICAgICAgIHJldHVybiB0aGlzLm1ldHJpYygnVENQX0NsaWVudF9SZXNldF9Db3VudCcsIHtcbiAgICAgICAgICAgIHN0YXRpc3RpYzogJ1N1bScsXG4gICAgICAgICAgICAuLi5wcm9wcyxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFRoZSB0b3RhbCBudW1iZXIgb2YgcmVzZXQgKFJTVCkgcGFja2V0cyBnZW5lcmF0ZWQgYnkgdGhlIGxvYWQgYmFsYW5jZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBTdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICAgKi9cbiAgICBwdWJsaWMgbWV0cmljVGNwRWxiUmVzZXRDb3VudChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucykge1xuICAgICAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ1RDUF9FTEJfUmVzZXRfQ291bnQnLCB7XG4gICAgICAgICAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgdG90YWwgbnVtYmVyIG9mIHJlc2V0IChSU1QpIHBhY2tldHMgc2VudCBmcm9tIGEgdGFyZ2V0IHRvIGEgY2xpZW50LlxuICAgICAqXG4gICAgICogVGhlc2UgcmVzZXRzIGFyZSBnZW5lcmF0ZWQgYnkgdGhlIHRhcmdldCBhbmQgZm9yd2FyZGVkIGJ5IHRoZSBsb2FkIGJhbGFuY2VyLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgU3VtIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpY1RjcFRhcmdldFJlc2V0Q291bnQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMubWV0cmljKCdUQ1BfVGFyZ2V0X1Jlc2V0X0NvdW50Jywge1xuICAgICAgICAgICAgc3RhdGlzdGljOiAnU3VtJyxcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICB9KTtcbiAgICB9XG59XG4vKipcbiAqIEEgbmV0d29yayBsb2FkIGJhbGFuY2VyXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSU5ldHdvcmtMb2FkQmFsYW5jZXIgZXh0ZW5kcyBJTG9hZEJhbGFuY2VyVjIsIGVjMi5JVnBjRW5kcG9pbnRTZXJ2aWNlTG9hZEJhbGFuY2VyIHtcbiAgICAvKipcbiAgICAgKiBUaGUgVlBDIHRoaXMgbG9hZCBiYWxhbmNlciBoYXMgYmVlbiBjcmVhdGVkIGluIChpZiBhdmFpbGFibGUpXG4gICAgICovXG4gICAgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG4gICAgLyoqXG4gICAgICogQWRkIGEgbGlzdGVuZXIgdG8gdGhpcyBsb2FkIGJhbGFuY2VyXG4gICAgICpcbiAgICAgKiBAcmV0dXJucyBUaGUgbmV3bHkgY3JlYXRlZCBsaXN0ZW5lclxuICAgICAqL1xuICAgIGFkZExpc3RlbmVyKGlkOiBzdHJpbmcsIHByb3BzOiBCYXNlTmV0d29ya0xpc3RlbmVyUHJvcHMpOiBOZXR3b3JrTGlzdGVuZXI7XG59XG4iXX0=