"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const application_listener_action_1 = require("./application-listener-action");
/**
 * The content type for a fixed response
 */
var ContentType;
(function (ContentType) {
    ContentType["TEXT_PLAIN"] = "text/plain";
    ContentType["TEXT_CSS"] = "text/css";
    ContentType["TEXT_HTML"] = "text/html";
    ContentType["APPLICATION_JAVASCRIPT"] = "application/javascript";
    ContentType["APPLICATION_JSON"] = "application/json";
})(ContentType = exports.ContentType || (exports.ContentType = {}));
/**
 * Define a new listener rule
 */
class ApplicationListenerRule extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.legacyConditions = {};
        this.conditions = props.conditions || [];
        const hasPathPatterns = props.pathPatterns || props.pathPattern;
        if (this.conditions.length === 0 && !props.hostHeader && !hasPathPatterns) {
            throw new Error('At least one of \'conditions\', \'hostHeader\', \'pathPattern\' or \'pathPatterns\' is required when defining a load balancing rule.');
        }
        const possibleActions = ['action', 'targetGroups', 'fixedResponse', 'redirectResponse'];
        const providedActions = possibleActions.filter(action => props[action] !== undefined);
        if (providedActions.length > 1) {
            throw new Error(`'${providedActions}' specified together, specify only one`);
        }
        if (props.priority <= 0) {
            throw new Error('Priority must have value greater than or equal to 1');
        }
        this.listener = props.listener;
        const resource = new elasticloadbalancingv2_generated_1.CfnListenerRule(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            priority: props.priority,
            conditions: cdk.Lazy.anyValue({ produce: () => this.renderConditions() }),
            actions: cdk.Lazy.anyValue({ produce: () => this.action ? this.action.renderActions() : [] }),
        });
        if (props.hostHeader) {
            this.setCondition('host-header', [props.hostHeader]);
        }
        if (hasPathPatterns) {
            if (props.pathPattern && props.pathPatterns) {
                throw new Error('Both `pathPatterns` and `pathPattern` are specified, specify only one');
            }
            const pathPattern = props.pathPattern ? [props.pathPattern] : props.pathPatterns;
            this.setCondition('path-pattern', pathPattern);
        }
        if (props.action) {
            this.configureAction(props.action);
        }
        (props.targetGroups || []).forEach(this.addTargetGroup.bind(this));
        if (props.fixedResponse) {
            this.addFixedResponse(props.fixedResponse);
        }
        else if (props.redirectResponse) {
            this.addRedirectResponse(props.redirectResponse);
        }
        this.listenerRuleArn = resource.ref;
    }
    /**
     * Add a non-standard condition to this rule
     *
     * If the condition conflicts with an already set condition, it will be overwritten by the one you specified.
     *
     * @deprecated use `addCondition` instead.
     */
    setCondition(field, values) {
        if (values === undefined) {
            delete this.legacyConditions[field];
            return;
        }
        this.legacyConditions[field] = values;
    }
    /**
     * Add a non-standard condition to this rule
     */
    addCondition(condition) {
        this.conditions.push(condition);
    }
    /**
     * Configure the action to perform for this rule
     */
    configureAction(action) {
        // It might make sense to 'throw' here.
        //
        // However, programs may already exist out there which configured an action twice,
        // in which case the second action accidentally overwrite the initial action, and in some
        // way ended up with a program that did what the author intended. If we were to add throw now,
        // the previously working program would be broken.
        //
        // Instead, signal this through a warning.
        // @deprecate: upon the next major version bump, replace this with a `throw`
        if (this.action) {
            this.node.addWarning('An Action already existed on this ListenerRule and was replaced. Configure exactly one default Action.');
        }
        action.bind(this, this.listener, this);
        this.action = action;
    }
    /**
     * Add a TargetGroup to load balance to
     *
     * @deprecated Use configureAction instead
     */
    addTargetGroup(targetGroup) {
        this.configureAction(application_listener_action_1.ListenerAction.forward([targetGroup]));
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use configureAction instead
     */
    addFixedResponse(fixedResponse) {
        validateFixedResponse(fixedResponse);
        this.configureAction(application_listener_action_1.ListenerAction.fixedResponse(cdk.Token.asNumber(fixedResponse.statusCode), {
            contentType: fixedResponse.contentType,
            messageBody: fixedResponse.messageBody,
        }));
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use configureAction instead
     */
    addRedirectResponse(redirectResponse) {
        validateRedirectResponse(redirectResponse);
        this.configureAction(application_listener_action_1.ListenerAction.redirect({
            host: redirectResponse.host,
            path: redirectResponse.path,
            permanent: redirectResponse.statusCode === 'HTTP_301',
            port: redirectResponse.port,
            protocol: redirectResponse.protocol,
            query: redirectResponse.query,
        }));
    }
    /**
     * Validate the rule
     */
    validate() {
        if (this.action === undefined) {
            return ['Listener rule needs at least one action'];
        }
        const legacyConditionFields = Object.keys(this.legacyConditions);
        if (legacyConditionFields.length === 0 && this.conditions.length === 0) {
            return ['Listener rule needs at least one condition'];
        }
        return [];
    }
    /**
     * Render the conditions for this rule
     */
    renderConditions() {
        const legacyConditions = Object.entries(this.legacyConditions).map(([field, values]) => {
            return { field, values };
        });
        const conditions = this.conditions.map(condition => condition.renderRawCondition());
        return [
            ...legacyConditions,
            ...conditions,
        ];
    }
}
exports.ApplicationListenerRule = ApplicationListenerRule;
/**
 * Validate the status code and message body of a fixed response
 *
 * @internal
 */
function validateFixedResponse(fixedResponse) {
    if (fixedResponse.statusCode && !/^(2|4|5)\d\d$/.test(fixedResponse.statusCode)) {
        throw new Error('`statusCode` must be 2XX, 4XX or 5XX.');
    }
    if (fixedResponse.messageBody && fixedResponse.messageBody.length > 1024) {
        throw new Error('`messageBody` cannot have more than 1024 characters.');
    }
}
exports.validateFixedResponse = validateFixedResponse;
/**
 * Validate the status code and message body of a redirect response
 *
 * @internal
 */
function validateRedirectResponse(redirectResponse) {
    if (redirectResponse.protocol && !/^(HTTPS?|#\{protocol\})$/i.test(redirectResponse.protocol)) {
        throw new Error('`protocol` must be HTTP, HTTPS, or #{protocol}.');
    }
    if (!redirectResponse.statusCode || !/^HTTP_30[12]$/.test(redirectResponse.statusCode)) {
        throw new Error('`statusCode` must be HTTP_301 or HTTP_302.');
    }
}
exports.validateRedirectResponse = validateRedirectResponse;
//# sourceMappingURL=data:application/json;base64,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