"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * What to do when a client makes a request to a listener
 *
 * Some actions can be combined with other ones (specifically,
 * you can perform authentication before serving the request).
 *
 * Multiple actions form a linked chain; the chain must always terminate in a
 * *(weighted)forward*, *fixedResponse* or *redirect* action.
 *
 * If an action supports chaining, the next action can be indicated
 * by passing it in the `next` property.
 *
 * (Called `ListenerAction` instead of the more strictly correct
 * `ListenerAction` because this is the class most users interact
 * with, and we want to make it not too visually overwhelming).
 */
class ListenerAction {
    /**
     * Create an instance of ListenerAction
     *
     * The default class should be good enough for most cases and
     * should be created by using one of the static factory functions,
     * but allow overriding to make sure we allow flexibility for the future.
     */
    constructor(actionJson, next) {
        this.actionJson = actionJson;
        this.next = next;
    }
    /**
     * Authenticate using an identity provider (IdP) that is compliant with OpenID Connect (OIDC)
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-authenticate-users.html#oidc-requirements
     */
    static authenticateOidc(options) {
        var _a;
        return new ListenerAction({
            type: 'authenticate-oidc',
            authenticateOidcConfig: {
                authorizationEndpoint: options.authorizationEndpoint,
                clientId: options.clientId,
                clientSecret: options.clientSecret.toString(),
                issuer: options.issuer,
                tokenEndpoint: options.tokenEndpoint,
                userInfoEndpoint: options.userInfoEndpoint,
                authenticationRequestExtraParams: options.authenticationRequestExtraParams,
                onUnauthenticatedRequest: options.onUnauthenticatedRequest,
                scope: options.scope,
                sessionCookieName: options.sessionCookieName,
                sessionTimeout: (_a = options.sessionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            },
        }, options.next);
    }
    /**
     * Forward to one or more Target Groups
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     */
    static forward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a ListenerAction.forward()');
        }
        if (targetGroups.length === 1 && options.stickinessDuration === undefined) {
            // Render a "simple" action for backwards compatibility with old templates
            return new TargetGroupListenerAction(targetGroups, {
                type: 'forward',
                targetGroupArn: targetGroups[0].targetGroupArn,
            });
        }
        return new TargetGroupListenerAction(targetGroups, {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroupArn })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Forward to one or more Target Groups which are weighted differently
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     */
    static weightedForward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a ListenerAction.weightedForward()');
        }
        return new TargetGroupListenerAction(targetGroups.map(g => g.targetGroup), {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroup.targetGroupArn, weight: g.weight })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Return a fixed response
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#fixed-response-actions
     */
    static fixedResponse(statusCode, options = {}) {
        return new ListenerAction({
            type: 'fixed-response',
            fixedResponseConfig: {
                statusCode: core_1.Tokenization.stringifyNumber(statusCode),
                contentType: options.contentType,
                messageBody: options.messageBody,
            },
        });
    }
    /**
     * Redirect to a different URI
     *
     * A URI consists of the following components:
     * protocol://hostname:port/path?query. You must modify at least one of the
     * following components to avoid a redirect loop: protocol, hostname, port, or
     * path. Any components that you do not modify retain their original values.
     *
     * You can reuse URI components using the following reserved keywords:
     *
     * - `#{protocol}`
     * - `#{host}`
     * - `#{port}`
     * - `#{path}` (the leading "/" is removed)
     * - `#{query}`
     *
     * For example, you can change the path to "/new/#{path}", the hostname to
     * "example.#{host}", or the query to "#{query}&value=xyz".
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#redirect-actions
     */
    static redirect(options) {
        if ([options.host, options.path, options.port, options.protocol, options.query].findIndex(x => x !== undefined) === -1) {
            throw new Error('To prevent redirect loops, set at least one of \'protocol\', \'host\', \'port\', \'path\', or \'query\'.');
        }
        return new ListenerAction({
            type: 'redirect',
            redirectConfig: {
                statusCode: options.permanent ? 'HTTP_301' : 'HTTP_302',
                host: options.host,
                path: options.path,
                port: options.port,
                protocol: options.protocol,
                query: options.query,
            },
        });
    }
    /**
     * Render the actions in this chain
     */
    renderActions() {
        var _a, _b;
        return this.renumber([this.actionJson, ...(_b = (_a = this.next) === null || _a === void 0 ? void 0 : _a.renderActions()) !== null && _b !== void 0 ? _b : []]);
    }
    /**
     * Called when the action is being used in a listener
     */
    bind(scope, listener, associatingConstruct) {
        // Empty on purpose
        Array.isArray(scope);
        Array.isArray(listener);
        Array.isArray(associatingConstruct);
    }
    /**
     * Renumber the "order" fields in the actions array.
     *
     * We don't number for 0 or 1 elements, but otherwise number them 1...#actions
     * so ELB knows about the right order.
     *
     * Do this in `ListenerAction` instead of in `Listener` so that we give
     * users the opportunity to override by subclassing and overriding `renderActions`.
     */
    renumber(actions) {
        if (actions.length < 2) {
            return actions;
        }
        return actions.map((action, i) => ({ ...action, order: i + 1 }));
    }
}
exports.ListenerAction = ListenerAction;
/**
 * What to do with unauthenticated requests
 */
var UnauthenticatedAction;
(function (UnauthenticatedAction) {
    /**
     * Return an HTTP 401 Unauthorized error.
     */
    UnauthenticatedAction["DENY"] = "deny";
    /**
     * Allow the request to be forwarded to the target.
     */
    UnauthenticatedAction["ALLOW"] = "allow";
    /**
     * Redirect the request to the IdP authorization endpoint.
     */
    UnauthenticatedAction["AUTHENTICATE"] = "authenticate";
})(UnauthenticatedAction = exports.UnauthenticatedAction || (exports.UnauthenticatedAction = {}));
/**
 * Listener Action that calls "registerListener" on TargetGroups
 */
class TargetGroupListenerAction extends ListenerAction {
    constructor(targetGroups, actionJson) {
        super(actionJson);
        this.targetGroups = targetGroups;
    }
    bind(_scope, listener, associatingConstruct) {
        for (const tg of this.targetGroups) {
            tg.registerListener(listener, associatingConstruct);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbGlzdGVuZXItYWN0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXBwbGljYXRpb24tbGlzdGVuZXItYWN0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQTJGLENBQUMsZ0RBQWdEO0FBSzVJOzs7Ozs7Ozs7Ozs7Ozs7R0FlRztBQUNILE1BQWEsY0FBYztJQTJIdkI7Ozs7OztPQU1HO0lBQ0gsWUFBdUMsVUFBc0MsRUFBcUIsSUFBcUI7UUFBaEYsZUFBVSxHQUFWLFVBQVUsQ0FBNEI7UUFBcUIsU0FBSSxHQUFKLElBQUksQ0FBaUI7SUFDdkgsQ0FBQztJQWxJRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLGdCQUFnQixDQUFDLE9BQWdDOztRQUMzRCxPQUFPLElBQUksY0FBYyxDQUFDO1lBQ3RCLElBQUksRUFBRSxtQkFBbUI7WUFDekIsc0JBQXNCLEVBQUU7Z0JBQ3BCLHFCQUFxQixFQUFFLE9BQU8sQ0FBQyxxQkFBcUI7Z0JBQ3BELFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUTtnQkFDMUIsWUFBWSxFQUFFLE9BQU8sQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFO2dCQUM3QyxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU07Z0JBQ3RCLGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYTtnQkFDcEMsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDLGdCQUFnQjtnQkFDMUMsZ0NBQWdDLEVBQUUsT0FBTyxDQUFDLGdDQUFnQztnQkFDMUUsd0JBQXdCLEVBQUUsT0FBTyxDQUFDLHdCQUF3QjtnQkFDMUQsS0FBSyxFQUFFLE9BQU8sQ0FBQyxLQUFLO2dCQUNwQixpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCO2dCQUM1QyxjQUFjLFFBQUUsT0FBTyxDQUFDLGNBQWMsMENBQUUsU0FBUyxFQUFFO2FBQ3REO1NBQ0osRUFBRSxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDckIsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsT0FBTyxDQUFDLFlBQXVDLEVBQUUsVUFBMEIsRUFBRTtRQUN2RixJQUFJLFlBQVksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELENBQUMsQ0FBQztTQUNsRjtRQUNELElBQUksWUFBWSxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksT0FBTyxDQUFDLGtCQUFrQixLQUFLLFNBQVMsRUFBRTtZQUN2RSwwRUFBMEU7WUFDMUUsT0FBTyxJQUFJLHlCQUF5QixDQUFDLFlBQVksRUFBRTtnQkFDL0MsSUFBSSxFQUFFLFNBQVM7Z0JBQ2YsY0FBYyxFQUFFLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjO2FBQ2pELENBQUMsQ0FBQztTQUNOO1FBQ0QsT0FBTyxJQUFJLHlCQUF5QixDQUFDLFlBQVksRUFBRTtZQUMvQyxJQUFJLEVBQUUsU0FBUztZQUNmLGFBQWEsRUFBRTtnQkFDWCxZQUFZLEVBQUUsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUM7Z0JBQzNFLDJCQUEyQixFQUFFLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7b0JBQ3RELGVBQWUsRUFBRSxPQUFPLENBQUMsa0JBQWtCLENBQUMsU0FBUyxFQUFFO29CQUN2RCxPQUFPLEVBQUUsSUFBSTtpQkFDaEIsQ0FBQyxDQUFDLENBQUMsU0FBUzthQUNoQjtTQUNKLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLGVBQWUsQ0FBQyxZQUFtQyxFQUFFLFVBQTBCLEVBQUU7UUFDM0YsSUFBSSxZQUFZLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUMzQixNQUFNLElBQUksS0FBSyxDQUFDLHFFQUFxRSxDQUFDLENBQUM7U0FDMUY7UUFDRCxPQUFPLElBQUkseUJBQXlCLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUN2RSxJQUFJLEVBQUUsU0FBUztZQUNmLGFBQWEsRUFBRTtnQkFDWCxZQUFZLEVBQUUsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxjQUFjLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO2dCQUN6RywyQkFBMkIsRUFBRSxPQUFPLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO29CQUN0RCxlQUFlLEVBQUUsT0FBTyxDQUFDLGtCQUFrQixDQUFDLFNBQVMsRUFBRTtvQkFDdkQsT0FBTyxFQUFFLElBQUk7aUJBQ2hCLENBQUMsQ0FBQyxDQUFDLFNBQVM7YUFDaEI7U0FDSixDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxhQUFhLENBQUMsVUFBa0IsRUFBRSxVQUFnQyxFQUFFO1FBQzlFLE9BQU8sSUFBSSxjQUFjLENBQUM7WUFDdEIsSUFBSSxFQUFFLGdCQUFnQjtZQUN0QixtQkFBbUIsRUFBRTtnQkFDakIsVUFBVSxFQUFFLG1CQUFZLENBQUMsZUFBZSxDQUFDLFVBQVUsQ0FBQztnQkFDcEQsV0FBVyxFQUFFLE9BQU8sQ0FBQyxXQUFXO2dCQUNoQyxXQUFXLEVBQUUsT0FBTyxDQUFDLFdBQVc7YUFDbkM7U0FDSixDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bb0JHO0lBQ0ksTUFBTSxDQUFDLFFBQVEsQ0FBQyxPQUF3QjtRQUMzQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO1lBQ3BILE1BQU0sSUFBSSxLQUFLLENBQUMsMEdBQTBHLENBQUMsQ0FBQztTQUMvSDtRQUNELE9BQU8sSUFBSSxjQUFjLENBQUM7WUFDdEIsSUFBSSxFQUFFLFVBQVU7WUFDaEIsY0FBYyxFQUFFO2dCQUNaLFVBQVUsRUFBRSxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFVBQVU7Z0JBQ3ZELElBQUksRUFBRSxPQUFPLENBQUMsSUFBSTtnQkFDbEIsSUFBSSxFQUFFLE9BQU8sQ0FBQyxJQUFJO2dCQUNsQixJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUk7Z0JBQ2xCLFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUTtnQkFDMUIsS0FBSyxFQUFFLE9BQU8sQ0FBQyxLQUFLO2FBQ3ZCO1NBQ0osQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQVVEOztPQUVHO0lBQ0ksYUFBYTs7UUFDaEIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxlQUFHLElBQUksQ0FBQyxJQUFJLDBDQUFFLGFBQWEscUNBQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBQ0Q7O09BRUc7SUFDSSxJQUFJLENBQUMsS0FBZ0IsRUFBRSxRQUE4QixFQUFFLG9CQUFpQztRQUMzRixtQkFBbUI7UUFDbkIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNyQixLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3hCLEtBQUssQ0FBQyxPQUFPLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBQ0Q7Ozs7Ozs7O09BUUc7SUFDTyxRQUFRLENBQUMsT0FBcUM7UUFDcEQsSUFBSSxPQUFPLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNwQixPQUFPLE9BQU8sQ0FBQztTQUNsQjtRQUNELE9BQU8sT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsRUFBRSxHQUFHLE1BQU0sRUFBRSxLQUFLLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNyRSxDQUFDO0NBQ0o7QUFsS0Qsd0NBa0tDO0FBZ01EOztHQUVHO0FBQ0gsSUFBWSxxQkFhWDtBQWJELFdBQVkscUJBQXFCO0lBQzdCOztPQUVHO0lBQ0gsc0NBQWEsQ0FBQTtJQUNiOztPQUVHO0lBQ0gsd0NBQWUsQ0FBQTtJQUNmOztPQUVHO0lBQ0gsc0RBQTZCLENBQUE7QUFDakMsQ0FBQyxFQWJXLHFCQUFxQixHQUFyQiw2QkFBcUIsS0FBckIsNkJBQXFCLFFBYWhDO0FBQ0Q7O0dBRUc7QUFDSCxNQUFNLHlCQUEwQixTQUFRLGNBQWM7SUFDbEQsWUFBNkIsWUFBdUMsRUFBRSxVQUFzQztRQUN4RyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7UUFETyxpQkFBWSxHQUFaLFlBQVksQ0FBMkI7SUFFcEUsQ0FBQztJQUNNLElBQUksQ0FBQyxNQUFpQixFQUFFLFFBQThCLEVBQUUsb0JBQWlDO1FBQzVGLEtBQUssTUFBTSxFQUFFLElBQUksSUFBSSxDQUFDLFlBQVksRUFBRTtZQUNoQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxFQUFFLG9CQUFvQixDQUFDLENBQUM7U0FDdkQ7SUFDTCxDQUFDO0NBQ0oiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb25zdHJ1Y3QsIER1cmF0aW9uLCBJQ29uc3RydWN0LCBTZWNyZXRWYWx1ZSwgVG9rZW5pemF0aW9uIH0gZnJvbSBcIi4uLy4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBDZm5MaXN0ZW5lciB9IGZyb20gJy4uL2VsYXN0aWNsb2FkYmFsYW5jaW5ndjIuZ2VuZXJhdGVkJztcbmltcG9ydCB7IElMaXN0ZW5lckFjdGlvbiB9IGZyb20gJy4uL3NoYXJlZC9saXN0ZW5lci1hY3Rpb24nO1xuaW1wb3J0IHsgSUFwcGxpY2F0aW9uTGlzdGVuZXIgfSBmcm9tICcuL2FwcGxpY2F0aW9uLWxpc3RlbmVyJztcbmltcG9ydCB7IElBcHBsaWNhdGlvblRhcmdldEdyb3VwIH0gZnJvbSAnLi9hcHBsaWNhdGlvbi10YXJnZXQtZ3JvdXAnO1xuLyoqXG4gKiBXaGF0IHRvIGRvIHdoZW4gYSBjbGllbnQgbWFrZXMgYSByZXF1ZXN0IHRvIGEgbGlzdGVuZXJcbiAqXG4gKiBTb21lIGFjdGlvbnMgY2FuIGJlIGNvbWJpbmVkIHdpdGggb3RoZXIgb25lcyAoc3BlY2lmaWNhbGx5LFxuICogeW91IGNhbiBwZXJmb3JtIGF1dGhlbnRpY2F0aW9uIGJlZm9yZSBzZXJ2aW5nIHRoZSByZXF1ZXN0KS5cbiAqXG4gKiBNdWx0aXBsZSBhY3Rpb25zIGZvcm0gYSBsaW5rZWQgY2hhaW47IHRoZSBjaGFpbiBtdXN0IGFsd2F5cyB0ZXJtaW5hdGUgaW4gYVxuICogKih3ZWlnaHRlZClmb3J3YXJkKiwgKmZpeGVkUmVzcG9uc2UqIG9yICpyZWRpcmVjdCogYWN0aW9uLlxuICpcbiAqIElmIGFuIGFjdGlvbiBzdXBwb3J0cyBjaGFpbmluZywgdGhlIG5leHQgYWN0aW9uIGNhbiBiZSBpbmRpY2F0ZWRcbiAqIGJ5IHBhc3NpbmcgaXQgaW4gdGhlIGBuZXh0YCBwcm9wZXJ0eS5cbiAqXG4gKiAoQ2FsbGVkIGBMaXN0ZW5lckFjdGlvbmAgaW5zdGVhZCBvZiB0aGUgbW9yZSBzdHJpY3RseSBjb3JyZWN0XG4gKiBgTGlzdGVuZXJBY3Rpb25gIGJlY2F1c2UgdGhpcyBpcyB0aGUgY2xhc3MgbW9zdCB1c2VycyBpbnRlcmFjdFxuICogd2l0aCwgYW5kIHdlIHdhbnQgdG8gbWFrZSBpdCBub3QgdG9vIHZpc3VhbGx5IG92ZXJ3aGVsbWluZykuXG4gKi9cbmV4cG9ydCBjbGFzcyBMaXN0ZW5lckFjdGlvbiBpbXBsZW1lbnRzIElMaXN0ZW5lckFjdGlvbiB7XG4gICAgLyoqXG4gICAgICogQXV0aGVudGljYXRlIHVzaW5nIGFuIGlkZW50aXR5IHByb3ZpZGVyIChJZFApIHRoYXQgaXMgY29tcGxpYW50IHdpdGggT3BlbklEIENvbm5lY3QgKE9JREMpXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljbG9hZGJhbGFuY2luZy9sYXRlc3QvYXBwbGljYXRpb24vbGlzdGVuZXItYXV0aGVudGljYXRlLXVzZXJzLmh0bWwjb2lkYy1yZXF1aXJlbWVudHNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGF1dGhlbnRpY2F0ZU9pZGMob3B0aW9uczogQXV0aGVudGljYXRlT2lkY09wdGlvbnMpOiBMaXN0ZW5lckFjdGlvbiB7XG4gICAgICAgIHJldHVybiBuZXcgTGlzdGVuZXJBY3Rpb24oe1xuICAgICAgICAgICAgdHlwZTogJ2F1dGhlbnRpY2F0ZS1vaWRjJyxcbiAgICAgICAgICAgIGF1dGhlbnRpY2F0ZU9pZGNDb25maWc6IHtcbiAgICAgICAgICAgICAgICBhdXRob3JpemF0aW9uRW5kcG9pbnQ6IG9wdGlvbnMuYXV0aG9yaXphdGlvbkVuZHBvaW50LFxuICAgICAgICAgICAgICAgIGNsaWVudElkOiBvcHRpb25zLmNsaWVudElkLFxuICAgICAgICAgICAgICAgIGNsaWVudFNlY3JldDogb3B0aW9ucy5jbGllbnRTZWNyZXQudG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICBpc3N1ZXI6IG9wdGlvbnMuaXNzdWVyLFxuICAgICAgICAgICAgICAgIHRva2VuRW5kcG9pbnQ6IG9wdGlvbnMudG9rZW5FbmRwb2ludCxcbiAgICAgICAgICAgICAgICB1c2VySW5mb0VuZHBvaW50OiBvcHRpb25zLnVzZXJJbmZvRW5kcG9pbnQsXG4gICAgICAgICAgICAgICAgYXV0aGVudGljYXRpb25SZXF1ZXN0RXh0cmFQYXJhbXM6IG9wdGlvbnMuYXV0aGVudGljYXRpb25SZXF1ZXN0RXh0cmFQYXJhbXMsXG4gICAgICAgICAgICAgICAgb25VbmF1dGhlbnRpY2F0ZWRSZXF1ZXN0OiBvcHRpb25zLm9uVW5hdXRoZW50aWNhdGVkUmVxdWVzdCxcbiAgICAgICAgICAgICAgICBzY29wZTogb3B0aW9ucy5zY29wZSxcbiAgICAgICAgICAgICAgICBzZXNzaW9uQ29va2llTmFtZTogb3B0aW9ucy5zZXNzaW9uQ29va2llTmFtZSxcbiAgICAgICAgICAgICAgICBzZXNzaW9uVGltZW91dDogb3B0aW9ucy5zZXNzaW9uVGltZW91dD8udG9TZWNvbmRzKCksXG4gICAgICAgICAgICB9LFxuICAgICAgICB9LCBvcHRpb25zLm5leHQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBGb3J3YXJkIHRvIG9uZSBvciBtb3JlIFRhcmdldCBHcm91cHNcbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNsb2FkYmFsYW5jaW5nL2xhdGVzdC9hcHBsaWNhdGlvbi9sb2FkLWJhbGFuY2VyLWxpc3RlbmVycy5odG1sI2ZvcndhcmQtYWN0aW9uc1xuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZm9yd2FyZCh0YXJnZXRHcm91cHM6IElBcHBsaWNhdGlvblRhcmdldEdyb3VwW10sIG9wdGlvbnM6IEZvcndhcmRPcHRpb25zID0ge30pOiBMaXN0ZW5lckFjdGlvbiB7XG4gICAgICAgIGlmICh0YXJnZXRHcm91cHMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ05lZWQgYXQgbGVhc3Qgb25lIHRhcmdldEdyb3VwIGluIGEgTGlzdGVuZXJBY3Rpb24uZm9yd2FyZCgpJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHRhcmdldEdyb3Vwcy5sZW5ndGggPT09IDEgJiYgb3B0aW9ucy5zdGlja2luZXNzRHVyYXRpb24gPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgLy8gUmVuZGVyIGEgXCJzaW1wbGVcIiBhY3Rpb24gZm9yIGJhY2t3YXJkcyBjb21wYXRpYmlsaXR5IHdpdGggb2xkIHRlbXBsYXRlc1xuICAgICAgICAgICAgcmV0dXJuIG5ldyBUYXJnZXRHcm91cExpc3RlbmVyQWN0aW9uKHRhcmdldEdyb3Vwcywge1xuICAgICAgICAgICAgICAgIHR5cGU6ICdmb3J3YXJkJyxcbiAgICAgICAgICAgICAgICB0YXJnZXRHcm91cEFybjogdGFyZ2V0R3JvdXBzWzBdLnRhcmdldEdyb3VwQXJuLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBUYXJnZXRHcm91cExpc3RlbmVyQWN0aW9uKHRhcmdldEdyb3Vwcywge1xuICAgICAgICAgICAgdHlwZTogJ2ZvcndhcmQnLFxuICAgICAgICAgICAgZm9yd2FyZENvbmZpZzoge1xuICAgICAgICAgICAgICAgIHRhcmdldEdyb3VwczogdGFyZ2V0R3JvdXBzLm1hcChnID0+ICh7IHRhcmdldEdyb3VwQXJuOiBnLnRhcmdldEdyb3VwQXJuIH0pKSxcbiAgICAgICAgICAgICAgICB0YXJnZXRHcm91cFN0aWNraW5lc3NDb25maWc6IG9wdGlvbnMuc3RpY2tpbmVzc0R1cmF0aW9uID8ge1xuICAgICAgICAgICAgICAgICAgICBkdXJhdGlvblNlY29uZHM6IG9wdGlvbnMuc3RpY2tpbmVzc0R1cmF0aW9uLnRvU2Vjb25kcygpLFxuICAgICAgICAgICAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgICAgIH0gOiB1bmRlZmluZWQsXG4gICAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogRm9yd2FyZCB0byBvbmUgb3IgbW9yZSBUYXJnZXQgR3JvdXBzIHdoaWNoIGFyZSB3ZWlnaHRlZCBkaWZmZXJlbnRseVxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItbGlzdGVuZXJzLmh0bWwjZm9yd2FyZC1hY3Rpb25zXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyB3ZWlnaHRlZEZvcndhcmQodGFyZ2V0R3JvdXBzOiBXZWlnaHRlZFRhcmdldEdyb3VwW10sIG9wdGlvbnM6IEZvcndhcmRPcHRpb25zID0ge30pOiBMaXN0ZW5lckFjdGlvbiB7XG4gICAgICAgIGlmICh0YXJnZXRHcm91cHMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ05lZWQgYXQgbGVhc3Qgb25lIHRhcmdldEdyb3VwIGluIGEgTGlzdGVuZXJBY3Rpb24ud2VpZ2h0ZWRGb3J3YXJkKCknKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gbmV3IFRhcmdldEdyb3VwTGlzdGVuZXJBY3Rpb24odGFyZ2V0R3JvdXBzLm1hcChnID0+IGcudGFyZ2V0R3JvdXApLCB7XG4gICAgICAgICAgICB0eXBlOiAnZm9yd2FyZCcsXG4gICAgICAgICAgICBmb3J3YXJkQ29uZmlnOiB7XG4gICAgICAgICAgICAgICAgdGFyZ2V0R3JvdXBzOiB0YXJnZXRHcm91cHMubWFwKGcgPT4gKHsgdGFyZ2V0R3JvdXBBcm46IGcudGFyZ2V0R3JvdXAudGFyZ2V0R3JvdXBBcm4sIHdlaWdodDogZy53ZWlnaHQgfSkpLFxuICAgICAgICAgICAgICAgIHRhcmdldEdyb3VwU3RpY2tpbmVzc0NvbmZpZzogb3B0aW9ucy5zdGlja2luZXNzRHVyYXRpb24gPyB7XG4gICAgICAgICAgICAgICAgICAgIGR1cmF0aW9uU2Vjb25kczogb3B0aW9ucy5zdGlja2luZXNzRHVyYXRpb24udG9TZWNvbmRzKCksXG4gICAgICAgICAgICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm4gYSBmaXhlZCByZXNwb25zZVxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItbGlzdGVuZXJzLmh0bWwjZml4ZWQtcmVzcG9uc2UtYWN0aW9uc1xuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZml4ZWRSZXNwb25zZShzdGF0dXNDb2RlOiBudW1iZXIsIG9wdGlvbnM6IEZpeGVkUmVzcG9uc2VPcHRpb25zID0ge30pOiBMaXN0ZW5lckFjdGlvbiB7XG4gICAgICAgIHJldHVybiBuZXcgTGlzdGVuZXJBY3Rpb24oe1xuICAgICAgICAgICAgdHlwZTogJ2ZpeGVkLXJlc3BvbnNlJyxcbiAgICAgICAgICAgIGZpeGVkUmVzcG9uc2VDb25maWc6IHtcbiAgICAgICAgICAgICAgICBzdGF0dXNDb2RlOiBUb2tlbml6YXRpb24uc3RyaW5naWZ5TnVtYmVyKHN0YXR1c0NvZGUpLFxuICAgICAgICAgICAgICAgIGNvbnRlbnRUeXBlOiBvcHRpb25zLmNvbnRlbnRUeXBlLFxuICAgICAgICAgICAgICAgIG1lc3NhZ2VCb2R5OiBvcHRpb25zLm1lc3NhZ2VCb2R5LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJlZGlyZWN0IHRvIGEgZGlmZmVyZW50IFVSSVxuICAgICAqXG4gICAgICogQSBVUkkgY29uc2lzdHMgb2YgdGhlIGZvbGxvd2luZyBjb21wb25lbnRzOlxuICAgICAqIHByb3RvY29sOi8vaG9zdG5hbWU6cG9ydC9wYXRoP3F1ZXJ5LiBZb3UgbXVzdCBtb2RpZnkgYXQgbGVhc3Qgb25lIG9mIHRoZVxuICAgICAqIGZvbGxvd2luZyBjb21wb25lbnRzIHRvIGF2b2lkIGEgcmVkaXJlY3QgbG9vcDogcHJvdG9jb2wsIGhvc3RuYW1lLCBwb3J0LCBvclxuICAgICAqIHBhdGguIEFueSBjb21wb25lbnRzIHRoYXQgeW91IGRvIG5vdCBtb2RpZnkgcmV0YWluIHRoZWlyIG9yaWdpbmFsIHZhbHVlcy5cbiAgICAgKlxuICAgICAqIFlvdSBjYW4gcmV1c2UgVVJJIGNvbXBvbmVudHMgdXNpbmcgdGhlIGZvbGxvd2luZyByZXNlcnZlZCBrZXl3b3JkczpcbiAgICAgKlxuICAgICAqIC0gYCN7cHJvdG9jb2x9YFxuICAgICAqIC0gYCN7aG9zdH1gXG4gICAgICogLSBgI3twb3J0fWBcbiAgICAgKiAtIGAje3BhdGh9YCAodGhlIGxlYWRpbmcgXCIvXCIgaXMgcmVtb3ZlZClcbiAgICAgKiAtIGAje3F1ZXJ5fWBcbiAgICAgKlxuICAgICAqIEZvciBleGFtcGxlLCB5b3UgY2FuIGNoYW5nZSB0aGUgcGF0aCB0byBcIi9uZXcvI3twYXRofVwiLCB0aGUgaG9zdG5hbWUgdG9cbiAgICAgKiBcImV4YW1wbGUuI3tob3N0fVwiLCBvciB0aGUgcXVlcnkgdG8gXCIje3F1ZXJ5fSZ2YWx1ZT14eXpcIi5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNsb2FkYmFsYW5jaW5nL2xhdGVzdC9hcHBsaWNhdGlvbi9sb2FkLWJhbGFuY2VyLWxpc3RlbmVycy5odG1sI3JlZGlyZWN0LWFjdGlvbnNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlZGlyZWN0KG9wdGlvbnM6IFJlZGlyZWN0T3B0aW9ucyk6IExpc3RlbmVyQWN0aW9uIHtcbiAgICAgICAgaWYgKFtvcHRpb25zLmhvc3QsIG9wdGlvbnMucGF0aCwgb3B0aW9ucy5wb3J0LCBvcHRpb25zLnByb3RvY29sLCBvcHRpb25zLnF1ZXJ5XS5maW5kSW5kZXgoeCA9PiB4ICE9PSB1bmRlZmluZWQpID09PSAtMSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUbyBwcmV2ZW50IHJlZGlyZWN0IGxvb3BzLCBzZXQgYXQgbGVhc3Qgb25lIG9mIFxcJ3Byb3RvY29sXFwnLCBcXCdob3N0XFwnLCBcXCdwb3J0XFwnLCBcXCdwYXRoXFwnLCBvciBcXCdxdWVyeVxcJy4nKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gbmV3IExpc3RlbmVyQWN0aW9uKHtcbiAgICAgICAgICAgIHR5cGU6ICdyZWRpcmVjdCcsXG4gICAgICAgICAgICByZWRpcmVjdENvbmZpZzoge1xuICAgICAgICAgICAgICAgIHN0YXR1c0NvZGU6IG9wdGlvbnMucGVybWFuZW50ID8gJ0hUVFBfMzAxJyA6ICdIVFRQXzMwMicsXG4gICAgICAgICAgICAgICAgaG9zdDogb3B0aW9ucy5ob3N0LFxuICAgICAgICAgICAgICAgIHBhdGg6IG9wdGlvbnMucGF0aCxcbiAgICAgICAgICAgICAgICBwb3J0OiBvcHRpb25zLnBvcnQsXG4gICAgICAgICAgICAgICAgcHJvdG9jb2w6IG9wdGlvbnMucHJvdG9jb2wsXG4gICAgICAgICAgICAgICAgcXVlcnk6IG9wdGlvbnMucXVlcnksXG4gICAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ3JlYXRlIGFuIGluc3RhbmNlIG9mIExpc3RlbmVyQWN0aW9uXG4gICAgICpcbiAgICAgKiBUaGUgZGVmYXVsdCBjbGFzcyBzaG91bGQgYmUgZ29vZCBlbm91Z2ggZm9yIG1vc3QgY2FzZXMgYW5kXG4gICAgICogc2hvdWxkIGJlIGNyZWF0ZWQgYnkgdXNpbmcgb25lIG9mIHRoZSBzdGF0aWMgZmFjdG9yeSBmdW5jdGlvbnMsXG4gICAgICogYnV0IGFsbG93IG92ZXJyaWRpbmcgdG8gbWFrZSBzdXJlIHdlIGFsbG93IGZsZXhpYmlsaXR5IGZvciB0aGUgZnV0dXJlLlxuICAgICAqL1xuICAgIHByb3RlY3RlZCBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGFjdGlvbkpzb246IENmbkxpc3RlbmVyLkFjdGlvblByb3BlcnR5LCBwcm90ZWN0ZWQgcmVhZG9ubHkgbmV4dD86IExpc3RlbmVyQWN0aW9uKSB7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJlbmRlciB0aGUgYWN0aW9ucyBpbiB0aGlzIGNoYWluXG4gICAgICovXG4gICAgcHVibGljIHJlbmRlckFjdGlvbnMoKTogQ2ZuTGlzdGVuZXIuQWN0aW9uUHJvcGVydHlbXSB7XG4gICAgICAgIHJldHVybiB0aGlzLnJlbnVtYmVyKFt0aGlzLmFjdGlvbkpzb24sIC4uLnRoaXMubmV4dD8ucmVuZGVyQWN0aW9ucygpID8/IFtdXSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIENhbGxlZCB3aGVuIHRoZSBhY3Rpb24gaXMgYmVpbmcgdXNlZCBpbiBhIGxpc3RlbmVyXG4gICAgICovXG4gICAgcHVibGljIGJpbmQoc2NvcGU6IENvbnN0cnVjdCwgbGlzdGVuZXI6IElBcHBsaWNhdGlvbkxpc3RlbmVyLCBhc3NvY2lhdGluZ0NvbnN0cnVjdD86IElDb25zdHJ1Y3QpIHtcbiAgICAgICAgLy8gRW1wdHkgb24gcHVycG9zZVxuICAgICAgICBBcnJheS5pc0FycmF5KHNjb3BlKTtcbiAgICAgICAgQXJyYXkuaXNBcnJheShsaXN0ZW5lcik7XG4gICAgICAgIEFycmF5LmlzQXJyYXkoYXNzb2NpYXRpbmdDb25zdHJ1Y3QpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZW51bWJlciB0aGUgXCJvcmRlclwiIGZpZWxkcyBpbiB0aGUgYWN0aW9ucyBhcnJheS5cbiAgICAgKlxuICAgICAqIFdlIGRvbid0IG51bWJlciBmb3IgMCBvciAxIGVsZW1lbnRzLCBidXQgb3RoZXJ3aXNlIG51bWJlciB0aGVtIDEuLi4jYWN0aW9uc1xuICAgICAqIHNvIEVMQiBrbm93cyBhYm91dCB0aGUgcmlnaHQgb3JkZXIuXG4gICAgICpcbiAgICAgKiBEbyB0aGlzIGluIGBMaXN0ZW5lckFjdGlvbmAgaW5zdGVhZCBvZiBpbiBgTGlzdGVuZXJgIHNvIHRoYXQgd2UgZ2l2ZVxuICAgICAqIHVzZXJzIHRoZSBvcHBvcnR1bml0eSB0byBvdmVycmlkZSBieSBzdWJjbGFzc2luZyBhbmQgb3ZlcnJpZGluZyBgcmVuZGVyQWN0aW9uc2AuXG4gICAgICovXG4gICAgcHJvdGVjdGVkIHJlbnVtYmVyKGFjdGlvbnM6IENmbkxpc3RlbmVyLkFjdGlvblByb3BlcnR5W10pOiBDZm5MaXN0ZW5lci5BY3Rpb25Qcm9wZXJ0eVtdIHtcbiAgICAgICAgaWYgKGFjdGlvbnMubGVuZ3RoIDwgMikge1xuICAgICAgICAgICAgcmV0dXJuIGFjdGlvbnM7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIGFjdGlvbnMubWFwKChhY3Rpb24sIGkpID0+ICh7IC4uLmFjdGlvbiwgb3JkZXI6IGkgKyAxIH0pKTtcbiAgICB9XG59XG4vKipcbiAqIE9wdGlvbnMgZm9yIGBMaXN0ZW5lckFjdGlvbi5mb3J3YXJkKClgXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRm9yd2FyZE9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIEZvciBob3cgbG9uZyBjbGllbnRzIHNob3VsZCBiZSBkaXJlY3RlZCB0byB0aGUgc2FtZSB0YXJnZXQgZ3JvdXBcbiAgICAgKlxuICAgICAqIFJhbmdlIGJldHdlZW4gMSBzZWNvbmQgYW5kIDcgZGF5cy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gc3RpY2tpbmVzc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHN0aWNraW5lc3NEdXJhdGlvbj86IER1cmF0aW9uO1xufVxuLyoqXG4gKiBBIFRhcmdldCBHcm91cCBhbmQgd2VpZ2h0IGNvbWJpbmF0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgV2VpZ2h0ZWRUYXJnZXRHcm91cCB7XG4gICAgLyoqXG4gICAgICogVGhlIHRhcmdldCBncm91cFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRhcmdldEdyb3VwOiBJQXBwbGljYXRpb25UYXJnZXRHcm91cDtcbiAgICAvKipcbiAgICAgKiBUaGUgdGFyZ2V0IGdyb3VwJ3Mgd2VpZ2h0XG4gICAgICpcbiAgICAgKiBSYW5nZSBpcyBbMC4uMTAwMCkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAxXG4gICAgICovXG4gICAgcmVhZG9ubHkgd2VpZ2h0PzogbnVtYmVyO1xufVxuLyoqXG4gKiBPcHRpb25zIGZvciBgTGlzdGVuZXJBY3Rpb24uZml4ZWRSZXNwb25zZSgpYFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEZpeGVkUmVzcG9uc2VPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBDb250ZW50IFR5cGUgb2YgdGhlIHJlc3BvbnNlXG4gICAgICpcbiAgICAgKiBWYWxpZCBWYWx1ZXM6IHRleHQvcGxhaW4gfCB0ZXh0L2NzcyB8IHRleHQvaHRtbCB8IGFwcGxpY2F0aW9uL2phdmFzY3JpcHQgfCBhcHBsaWNhdGlvbi9qc29uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEF1dG9tYXRpY2FsbHkgZGV0ZXJtaW5lZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbnRlbnRUeXBlPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSByZXNwb25zZSBib2R5XG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGJvZHlcbiAgICAgKi9cbiAgICByZWFkb25seSBtZXNzYWdlQm9keT86IHN0cmluZztcbn1cbi8qKlxuICogT3B0aW9ucyBmb3IgYExpc3RlbmVyQWN0aW9uLnJlZGlyZWN0KClgXG4gKlxuICogQSBVUkkgY29uc2lzdHMgb2YgdGhlIGZvbGxvd2luZyBjb21wb25lbnRzOlxuICogcHJvdG9jb2w6Ly9ob3N0bmFtZTpwb3J0L3BhdGg/cXVlcnkuIFlvdSBtdXN0IG1vZGlmeSBhdCBsZWFzdCBvbmUgb2YgdGhlXG4gKiBmb2xsb3dpbmcgY29tcG9uZW50cyB0byBhdm9pZCBhIHJlZGlyZWN0IGxvb3A6IHByb3RvY29sLCBob3N0bmFtZSwgcG9ydCwgb3JcbiAqIHBhdGguIEFueSBjb21wb25lbnRzIHRoYXQgeW91IGRvIG5vdCBtb2RpZnkgcmV0YWluIHRoZWlyIG9yaWdpbmFsIHZhbHVlcy5cbiAqXG4gKiBZb3UgY2FuIHJldXNlIFVSSSBjb21wb25lbnRzIHVzaW5nIHRoZSBmb2xsb3dpbmcgcmVzZXJ2ZWQga2V5d29yZHM6XG4gKlxuICogLSBgI3twcm90b2NvbH1gXG4gKiAtIGAje2hvc3R9YFxuICogLSBgI3twb3J0fWBcbiAqIC0gYCN7cGF0aH1gICh0aGUgbGVhZGluZyBcIi9cIiBpcyByZW1vdmVkKVxuICogLSBgI3txdWVyeX1gXG4gKlxuICogRm9yIGV4YW1wbGUsIHlvdSBjYW4gY2hhbmdlIHRoZSBwYXRoIHRvIFwiL25ldy8je3BhdGh9XCIsIHRoZSBob3N0bmFtZSB0b1xuICogXCJleGFtcGxlLiN7aG9zdH1cIiwgb3IgdGhlIHF1ZXJ5IHRvIFwiI3txdWVyeX0mdmFsdWU9eHl6XCIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVkaXJlY3RPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBUaGUgaG9zdG5hbWUuXG4gICAgICpcbiAgICAgKiBUaGlzIGNvbXBvbmVudCBpcyBub3QgcGVyY2VudC1lbmNvZGVkLiBUaGUgaG9zdG5hbWUgY2FuIGNvbnRhaW4gI3tob3N0fS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gY2hhbmdlXG4gICAgICovXG4gICAgcmVhZG9ubHkgaG9zdD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgYWJzb2x1dGUgcGF0aCwgc3RhcnRpbmcgd2l0aCB0aGUgbGVhZGluZyBcIi9cIi5cbiAgICAgKlxuICAgICAqIFRoaXMgY29tcG9uZW50IGlzIG5vdCBwZXJjZW50LWVuY29kZWQuIFRoZSBwYXRoIGNhbiBjb250YWluICN7aG9zdH0sICN7cGF0aH0sIGFuZCAje3BvcnR9LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBjaGFuZ2VcbiAgICAgKi9cbiAgICByZWFkb25seSBwYXRoPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBwb3J0LlxuICAgICAqXG4gICAgICogWW91IGNhbiBzcGVjaWZ5IGEgdmFsdWUgZnJvbSAxIHRvIDY1NTM1IG9yICN7cG9ydH0uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGNoYW5nZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHBvcnQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHByb3RvY29sLlxuICAgICAqXG4gICAgICogWW91IGNhbiBzcGVjaWZ5IEhUVFAsIEhUVFBTLCBvciAje3Byb3RvY29sfS4gWW91IGNhbiByZWRpcmVjdCBIVFRQIHRvIEhUVFAsIEhUVFAgdG8gSFRUUFMsIGFuZCBIVFRQUyB0byBIVFRQUy4gWW91IGNhbm5vdCByZWRpcmVjdCBIVFRQUyB0byBIVFRQLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBjaGFuZ2VcbiAgICAgKi9cbiAgICByZWFkb25seSBwcm90b2NvbD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgcXVlcnkgcGFyYW1ldGVycywgVVJMLWVuY29kZWQgd2hlbiBuZWNlc3NhcnksIGJ1dCBub3QgcGVyY2VudC1lbmNvZGVkLlxuICAgICAqXG4gICAgICogRG8gbm90IGluY2x1ZGUgdGhlIGxlYWRpbmcgXCI/XCIsIGFzIGl0IGlzIGF1dG9tYXRpY2FsbHkgYWRkZWQuIFlvdSBjYW4gc3BlY2lmeSBhbnkgb2YgdGhlIHJlc2VydmVkIGtleXdvcmRzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBjaGFuZ2VcbiAgICAgKi9cbiAgICByZWFkb25seSBxdWVyeT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgSFRUUCByZWRpcmVjdCBjb2RlLlxuICAgICAqXG4gICAgICogVGhlIHJlZGlyZWN0IGlzIGVpdGhlciBwZXJtYW5lbnQgKEhUVFAgMzAxKSBvciB0ZW1wb3JhcnkgKEhUVFAgMzAyKS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGVybWFuZW50PzogYm9vbGVhbjtcbn1cbi8qKlxuICogT3B0aW9ucyBmb3IgYExpc3RlbmVyQWN0aW9uLmF1dGhlbmNpYXRlT2lkYygpYFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF1dGhlbnRpY2F0ZU9pZGNPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBXaGF0IGFjdGlvbiB0byBleGVjdXRlIG5leHRcbiAgICAgKi9cbiAgICByZWFkb25seSBuZXh0OiBMaXN0ZW5lckFjdGlvbjtcbiAgICAvKipcbiAgICAgKiBUaGUgcXVlcnkgcGFyYW1ldGVycyAodXAgdG8gMTApIHRvIGluY2x1ZGUgaW4gdGhlIHJlZGlyZWN0IHJlcXVlc3QgdG8gdGhlIGF1dGhvcml6YXRpb24gZW5kcG9pbnQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGV4dHJhIHBhcmFtZXRlcnNcbiAgICAgKi9cbiAgICByZWFkb25seSBhdXRoZW50aWNhdGlvblJlcXVlc3RFeHRyYVBhcmFtcz86IFJlY29yZDxzdHJpbmcsIHN0cmluZz47XG4gICAgLyoqXG4gICAgICogVGhlIGF1dGhvcml6YXRpb24gZW5kcG9pbnQgb2YgdGhlIElkUC5cbiAgICAgKlxuICAgICAqIFRoaXMgbXVzdCBiZSBhIGZ1bGwgVVJMLCBpbmNsdWRpbmcgdGhlIEhUVFBTIHByb3RvY29sLCB0aGUgZG9tYWluLCBhbmQgdGhlIHBhdGguXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXV0aG9yaXphdGlvbkVuZHBvaW50OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIE9BdXRoIDIuMCBjbGllbnQgaWRlbnRpZmllci5cbiAgICAgKi9cbiAgICByZWFkb25seSBjbGllbnRJZDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBPQXV0aCAyLjAgY2xpZW50IHNlY3JldC5cbiAgICAgKi9cbiAgICByZWFkb25seSBjbGllbnRTZWNyZXQ6IFNlY3JldFZhbHVlO1xuICAgIC8qKlxuICAgICAqIFRoZSBPSURDIGlzc3VlciBpZGVudGlmaWVyIG9mIHRoZSBJZFAuXG4gICAgICpcbiAgICAgKiBUaGlzIG11c3QgYmUgYSBmdWxsIFVSTCwgaW5jbHVkaW5nIHRoZSBIVFRQUyBwcm90b2NvbCwgdGhlIGRvbWFpbiwgYW5kIHRoZSBwYXRoLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGlzc3Vlcjogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBiZWhhdmlvciBpZiB0aGUgdXNlciBpcyBub3QgYXV0aGVudGljYXRlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFVuYXV0aGVudGljYXRlZEFjdGlvbi5BVVRIRU5USUNBVEVcbiAgICAgKi9cbiAgICByZWFkb25seSBvblVuYXV0aGVudGljYXRlZFJlcXVlc3Q/OiBVbmF1dGhlbnRpY2F0ZWRBY3Rpb247XG4gICAgLyoqXG4gICAgICogVGhlIHNldCBvZiB1c2VyIGNsYWltcyB0byBiZSByZXF1ZXN0ZWQgZnJvbSB0aGUgSWRQLlxuICAgICAqXG4gICAgICogVG8gdmVyaWZ5IHdoaWNoIHNjb3BlIHZhbHVlcyB5b3VyIElkUCBzdXBwb3J0cyBhbmQgaG93IHRvIHNlcGFyYXRlIG11bHRpcGxlIHZhbHVlcywgc2VlIHRoZSBkb2N1bWVudGF0aW9uIGZvciB5b3VyIElkUC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFwib3BlbmlkXCJcbiAgICAgKi9cbiAgICByZWFkb25seSBzY29wZT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgY29va2llIHVzZWQgdG8gbWFpbnRhaW4gc2Vzc2lvbiBpbmZvcm1hdGlvbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFwiQVdTRUxCQXV0aFNlc3Npb25Db29raWVcIlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNlc3Npb25Db29raWVOYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBtYXhpbXVtIGR1cmF0aW9uIG9mIHRoZSBhdXRoZW50aWNhdGlvbiBzZXNzaW9uLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRHVyYXRpb24uZGF5cyg3KVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNlc3Npb25UaW1lb3V0PzogRHVyYXRpb247XG4gICAgLyoqXG4gICAgICogVGhlIHRva2VuIGVuZHBvaW50IG9mIHRoZSBJZFAuXG4gICAgICpcbiAgICAgKiBUaGlzIG11c3QgYmUgYSBmdWxsIFVSTCwgaW5jbHVkaW5nIHRoZSBIVFRQUyBwcm90b2NvbCwgdGhlIGRvbWFpbiwgYW5kIHRoZSBwYXRoLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRva2VuRW5kcG9pbnQ6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgdXNlciBpbmZvIGVuZHBvaW50IG9mIHRoZSBJZFAuXG4gICAgICpcbiAgICAgKiBUaGlzIG11c3QgYmUgYSBmdWxsIFVSTCwgaW5jbHVkaW5nIHRoZSBIVFRQUyBwcm90b2NvbCwgdGhlIGRvbWFpbiwgYW5kIHRoZSBwYXRoLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHVzZXJJbmZvRW5kcG9pbnQ6IHN0cmluZztcbn1cbi8qKlxuICogV2hhdCB0byBkbyB3aXRoIHVuYXV0aGVudGljYXRlZCByZXF1ZXN0c1xuICovXG5leHBvcnQgZW51bSBVbmF1dGhlbnRpY2F0ZWRBY3Rpb24ge1xuICAgIC8qKlxuICAgICAqIFJldHVybiBhbiBIVFRQIDQwMSBVbmF1dGhvcml6ZWQgZXJyb3IuXG4gICAgICovXG4gICAgREVOWSA9ICdkZW55JyxcbiAgICAvKipcbiAgICAgKiBBbGxvdyB0aGUgcmVxdWVzdCB0byBiZSBmb3J3YXJkZWQgdG8gdGhlIHRhcmdldC5cbiAgICAgKi9cbiAgICBBTExPVyA9ICdhbGxvdycsXG4gICAgLyoqXG4gICAgICogUmVkaXJlY3QgdGhlIHJlcXVlc3QgdG8gdGhlIElkUCBhdXRob3JpemF0aW9uIGVuZHBvaW50LlxuICAgICAqL1xuICAgIEFVVEhFTlRJQ0FURSA9ICdhdXRoZW50aWNhdGUnXG59XG4vKipcbiAqIExpc3RlbmVyIEFjdGlvbiB0aGF0IGNhbGxzIFwicmVnaXN0ZXJMaXN0ZW5lclwiIG9uIFRhcmdldEdyb3Vwc1xuICovXG5jbGFzcyBUYXJnZXRHcm91cExpc3RlbmVyQWN0aW9uIGV4dGVuZHMgTGlzdGVuZXJBY3Rpb24ge1xuICAgIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgdGFyZ2V0R3JvdXBzOiBJQXBwbGljYXRpb25UYXJnZXRHcm91cFtdLCBhY3Rpb25Kc29uOiBDZm5MaXN0ZW5lci5BY3Rpb25Qcm9wZXJ0eSkge1xuICAgICAgICBzdXBlcihhY3Rpb25Kc29uKTtcbiAgICB9XG4gICAgcHVibGljIGJpbmQoX3Njb3BlOiBDb25zdHJ1Y3QsIGxpc3RlbmVyOiBJQXBwbGljYXRpb25MaXN0ZW5lciwgYXNzb2NpYXRpbmdDb25zdHJ1Y3Q/OiBJQ29uc3RydWN0KSB7XG4gICAgICAgIGZvciAoY29uc3QgdGcgb2YgdGhpcy50YXJnZXRHcm91cHMpIHtcbiAgICAgICAgICAgIHRnLnJlZ2lzdGVyTGlzdGVuZXIobGlzdGVuZXIsIGFzc29jaWF0aW5nQ29uc3RydWN0KTtcbiAgICAgICAgfVxuICAgIH1cbn1cbiJdfQ==