"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const log_driver_1 = require("./log-driver");
const utils_1 = require("./utils");
/**
 * The type of compression the GELF driver uses to compress each log message.
 */
var GelfCompressionType;
(function (GelfCompressionType) {
    GelfCompressionType["GZIP"] = "gzip";
    GelfCompressionType["ZLIB"] = "zlib";
    GelfCompressionType["NONE"] = "none";
})(GelfCompressionType = exports.GelfCompressionType || (exports.GelfCompressionType = {}));
/**
 * A log driver that sends log information to journald Logs.
 */
class GelfLogDriver extends log_driver_1.LogDriver {
    /**
     * Constructs a new instance of the GelfLogDriver class.
     *
     * @param props the gelf log driver configuration options.
     */
    constructor(props) {
        super();
        this.props = props;
        // Validation
        if (props.compressionLevel) {
            utils_1.ensureInRange(props.compressionLevel, -1, 9);
        }
        if (props.tcpMaxReconnect) {
            utils_1.ensurePositiveInteger(props.tcpMaxReconnect);
        }
    }
    /**
     * Called when the log driver is configured on a container
     */
    bind(_scope, _containerDefinition) {
        return {
            logDriver: 'gelf',
            options: utils_1.stringifyOptions({
                'gelf-address': this.props.address,
                'gelf-compression-type': this.props.compressionType,
                'gelf-compression-level': this.props.compressionLevel,
                'gelf-tcp-max-reconnect': this.props.tcpMaxReconnect,
                'gelf-tcp-reconnect-delay': this.props.tcpReconnectDelay && this.props.tcpReconnectDelay.toSeconds(),
                ...utils_1.renderCommonLogDriverOptions(this.props),
            }),
        };
    }
}
exports.GelfLogDriver = GelfLogDriver;
//# sourceMappingURL=data:application/json;base64,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