"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_definition_1 = require("../base/task-definition");
/**
 * The details of a task definition run on a Fargate cluster.
 *
 * @resource AWS::ECS::TaskDefinition
 */
class FargateTaskDefinition extends task_definition_1.TaskDefinition {
    // NOTE: Until the fix to https://github.com/Microsoft/TypeScript/issues/26969 gets released,
    // we need to explicitly write the type here, as type deduction for enums won't lead to
    // the import being generated in the .d.ts file.
    /**
     * Constructs a new instance of the FargateTaskDefinition class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            cpu: props.cpu !== undefined ? core_1.Tokenization.stringifyNumber(props.cpu) : '256',
            memoryMiB: props.memoryLimitMiB !== undefined ? core_1.Tokenization.stringifyNumber(props.memoryLimitMiB) : '512',
            compatibility: task_definition_1.Compatibility.FARGATE,
            networkMode: task_definition_1.NetworkMode.AWS_VPC,
        });
        /**
         * The Docker networking mode to use for the containers in the task. Fargate tasks require the awsvpc network mode.
         */
        this.networkMode = task_definition_1.NetworkMode.AWS_VPC;
    }
    /**
     * Imports a task definition from the specified task definition ARN.
     */
    static fromFargateTaskDefinitionArn(scope, id, fargateTaskDefinitionArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.taskDefinitionArn = fargateTaskDefinitionArn;
                this.compatibility = task_definition_1.Compatibility.FARGATE;
                this.isEc2Compatible = false;
                this.isFargateCompatible = true;
            }
        }
        return new Import(scope, id);
    }
}
exports.FargateTaskDefinition = FargateTaskDefinition;
//# sourceMappingURL=data:application/json;base64,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