"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const task_definition_1 = require("./base/task-definition");
/**
 * A secret environment variable.
 */
class Secret {
    /**
     * Creates an environment variable value from a parameter stored in AWS
     * Systems Manager Parameter Store.
     */
    static fromSsmParameter(parameter) {
        return {
            arn: parameter.parameterArn,
            grantRead: grantee => parameter.grantRead(grantee),
        };
    }
    /**
     * Creates a environment variable value from a secret stored in AWS Secrets
     * Manager.
     *
     * @param secret the secret stored in AWS Secrets Manager
     * @param field the name of the field with the value that you want to set as
     * the environment variable value. Only values in JSON format are supported.
     * If you do not specify a JSON field, then the full content of the secret is
     * used.
     */
    static fromSecretsManager(secret, field) {
        return {
            arn: field ? `${secret.secretArn}:${field}::` : secret.secretArn,
            hasField: !!field,
            grantRead: grantee => secret.grantRead(grantee),
        };
    }
}
exports.Secret = Secret;
/**
 * A container definition is used in a task definition to describe the containers that are launched as part of a task.
 */
class ContainerDefinition extends cdk.Construct {
    /**
     * Constructs a new instance of the ContainerDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        /**
         * The mount points for data volumes in your container.
         */
        this.mountPoints = new Array();
        /**
         * The list of port mappings for the container. Port mappings allow containers to access ports
         * on the host container instance to send or receive traffic.
         */
        this.portMappings = new Array();
        /**
         * The data volumes to mount from another container in the same task definition.
         */
        this.volumesFrom = new Array();
        /**
         * An array of ulimits to set in the container.
         */
        this.ulimits = new Array();
        /**
         * An array dependencies defined for container startup and shutdown.
         */
        this.containerDependencies = new Array();
        /**
         * The configured container links
         */
        this.links = new Array();
        if (props.memoryLimitMiB !== undefined && props.memoryReservationMiB !== undefined) {
            if (props.memoryLimitMiB < props.memoryReservationMiB) {
                throw new Error('MemoryLimitMiB should not be less than MemoryReservationMiB.');
            }
        }
        this.essential = props.essential !== undefined ? props.essential : true;
        this.taskDefinition = props.taskDefinition;
        this.memoryLimitSpecified = props.memoryLimitMiB !== undefined || props.memoryReservationMiB !== undefined;
        this.linuxParameters = props.linuxParameters;
        this.containerName = this.node.id;
        this.imageConfig = props.image.bind(this, this);
        if (props.logging) {
            this.logDriverConfig = props.logging.bind(this, this);
        }
        props.taskDefinition._linkContainer(this);
        if (props.secrets) {
            this.secrets = [];
            for (const [name, secret] of Object.entries(props.secrets)) {
                if (this.taskDefinition.isFargateCompatible && secret.hasField) {
                    throw new Error(`Cannot specify secret JSON field for a task using the FARGATE launch type: '${name}' in container '${this.node.id}'`);
                }
                secret.grantRead(this.taskDefinition.obtainExecutionRole());
                this.secrets.push({
                    name,
                    valueFrom: secret.arn,
                });
            }
        }
    }
    /**
     * This method adds a link which allows containers to communicate with each other without the need for port mappings.
     *
     * This parameter is only supported if the task definition is using the bridge network mode.
     * Warning: The --link flag is a legacy feature of Docker. It may eventually be removed.
     */
    addLink(container, alias) {
        if (this.taskDefinition.networkMode !== task_definition_1.NetworkMode.BRIDGE) {
            throw new Error('You must use network mode Bridge to add container links.');
        }
        if (alias !== undefined) {
            this.links.push(`${container.containerName}:${alias}`);
        }
        else {
            this.links.push(`${container.containerName}`);
        }
    }
    /**
     * This method adds one or more mount points for data volumes to the container.
     */
    addMountPoints(...mountPoints) {
        this.mountPoints.push(...mountPoints);
    }
    /**
     * This method mounts temporary disk space to the container.
     *
     * This adds the correct container mountPoint and task definition volume.
     */
    addScratch(scratch) {
        const mountPoint = {
            containerPath: scratch.containerPath,
            readOnly: scratch.readOnly,
            sourceVolume: scratch.name,
        };
        const volume = {
            host: {
                sourcePath: scratch.sourcePath,
            },
            name: scratch.name,
        };
        this.taskDefinition.addVolume(volume);
        this.addMountPoints(mountPoint);
    }
    /**
     * This method adds one or more port mappings to the container.
     */
    addPortMappings(...portMappings) {
        this.portMappings.push(...portMappings.map(pm => {
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC || this.taskDefinition.networkMode === task_definition_1.NetworkMode.HOST) {
                if (pm.containerPort !== pm.hostPort && pm.hostPort !== undefined) {
                    throw new Error(`Host port (${pm.hostPort}) must be left out or equal to container port ${pm.containerPort} for network mode ${this.taskDefinition.networkMode}`);
                }
            }
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
                if (pm.hostPort === undefined) {
                    pm = {
                        ...pm,
                        hostPort: 0,
                    };
                }
            }
            return pm;
        }));
    }
    /**
     * This method adds one or more ulimits to the container.
     */
    addUlimits(...ulimits) {
        this.ulimits.push(...ulimits);
    }
    /**
     * This method adds one or more container dependencies to the container.
     */
    addContainerDependencies(...containerDependencies) {
        this.containerDependencies.push(...containerDependencies);
    }
    /**
     * This method adds one or more volumes to the container.
     */
    addVolumesFrom(...volumesFrom) {
        this.volumesFrom.push(...volumesFrom);
    }
    /**
     * This method adds the specified statement to the IAM task execution policy in the task definition.
     */
    addToExecutionPolicy(statement) {
        this.taskDefinition.addToExecutionRolePolicy(statement);
    }
    /**
     * Returns the host port for the requested container port if it exists
     */
    findPortMapping(containerPort, protocol) {
        for (const portMapping of this.portMappings) {
            const p = portMapping.protocol || Protocol.TCP;
            const c = portMapping.containerPort;
            if (c === containerPort && p === protocol) {
                return portMapping;
            }
        }
        return undefined;
    }
    /**
     * The inbound rules associated with the security group the task or service will use.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     */
    get ingressPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        if (defaultPortMapping.hostPort !== undefined && defaultPortMapping.hostPort !== 0) {
            return defaultPortMapping.hostPort;
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
            return 0;
        }
        return defaultPortMapping.containerPort;
    }
    /**
     * The port the container will listen on.
     */
    get containerPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        return defaultPortMapping.containerPort;
    }
    /**
     * Render this container definition to a CloudFormation object
     *
     * @param _taskDefinition [disable-awslint:ref-via-interface] (unused but kept to avoid breaking change)
     */
    renderContainerDefinition(_taskDefinition) {
        return {
            command: this.props.command,
            cpu: this.props.cpu,
            disableNetworking: this.props.disableNetworking,
            dependsOn: cdk.Lazy.anyValue({ produce: () => this.containerDependencies.map(renderContainerDependency) }, { omitEmptyArray: true }),
            dnsSearchDomains: this.props.dnsSearchDomains,
            dnsServers: this.props.dnsServers,
            dockerLabels: this.props.dockerLabels,
            dockerSecurityOptions: this.props.dockerSecurityOptions,
            entryPoint: this.props.entryPoint,
            essential: this.essential,
            hostname: this.props.hostname,
            image: this.imageConfig.imageName,
            memory: this.props.memoryLimitMiB,
            memoryReservation: this.props.memoryReservationMiB,
            mountPoints: cdk.Lazy.anyValue({ produce: () => this.mountPoints.map(renderMountPoint) }, { omitEmptyArray: true }),
            name: this.containerName,
            portMappings: cdk.Lazy.anyValue({ produce: () => this.portMappings.map(renderPortMapping) }, { omitEmptyArray: true }),
            privileged: this.props.privileged,
            readonlyRootFilesystem: this.props.readonlyRootFilesystem,
            repositoryCredentials: this.imageConfig.repositoryCredentials,
            startTimeout: this.props.startTimeout && this.props.startTimeout.toSeconds(),
            stopTimeout: this.props.stopTimeout && this.props.stopTimeout.toSeconds(),
            ulimits: cdk.Lazy.anyValue({ produce: () => this.ulimits.map(renderUlimit) }, { omitEmptyArray: true }),
            user: this.props.user,
            volumesFrom: cdk.Lazy.anyValue({ produce: () => this.volumesFrom.map(renderVolumeFrom) }, { omitEmptyArray: true }),
            workingDirectory: this.props.workingDirectory,
            logConfiguration: this.logDriverConfig,
            environment: this.props.environment && renderKV(this.props.environment, 'name', 'value'),
            secrets: this.secrets,
            extraHosts: this.props.extraHosts && renderKV(this.props.extraHosts, 'hostname', 'ipAddress'),
            healthCheck: this.props.healthCheck && renderHealthCheck(this.props.healthCheck),
            links: cdk.Lazy.listValue({ produce: () => this.links }, { omitEmpty: true }),
            linuxParameters: this.linuxParameters && this.linuxParameters.renderLinuxParameters(),
            resourceRequirements: (this.props.gpuCount !== undefined) ? renderResourceRequirements(this.props.gpuCount) : undefined,
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
function renderKV(env, keyName, valueName) {
    const ret = new Array();
    for (const [key, value] of Object.entries(env)) {
        ret.push({ [keyName]: key, [valueName]: value });
    }
    return ret;
}
function renderHealthCheck(hc) {
    return {
        command: getHealthCheckCommand(hc),
        interval: hc.interval != null ? hc.interval.toSeconds() : 30,
        retries: hc.retries !== undefined ? hc.retries : 3,
        startPeriod: hc.startPeriod && hc.startPeriod.toSeconds(),
        timeout: hc.timeout !== undefined ? hc.timeout.toSeconds() : 5,
    };
}
function getHealthCheckCommand(hc) {
    const cmd = hc.command;
    const hcCommand = new Array();
    if (cmd.length === 0) {
        throw new Error('At least one argument must be supplied for health check command.');
    }
    if (cmd.length === 1) {
        hcCommand.push('CMD-SHELL', cmd[0]);
        return hcCommand;
    }
    if (cmd[0] !== 'CMD' && cmd[0] !== 'CMD-SHELL') {
        hcCommand.push('CMD');
    }
    return hcCommand.concat(cmd);
}
function renderResourceRequirements(gpuCount) {
    if (gpuCount === 0) {
        return undefined;
    }
    return [{
            type: 'GPU',
            value: gpuCount.toString(),
        }];
}
/**
 * Type of resource to set a limit on
 */
var UlimitName;
(function (UlimitName) {
    UlimitName["CORE"] = "core";
    UlimitName["CPU"] = "cpu";
    UlimitName["DATA"] = "data";
    UlimitName["FSIZE"] = "fsize";
    UlimitName["LOCKS"] = "locks";
    UlimitName["MEMLOCK"] = "memlock";
    UlimitName["MSGQUEUE"] = "msgqueue";
    UlimitName["NICE"] = "nice";
    UlimitName["NOFILE"] = "nofile";
    UlimitName["NPROC"] = "nproc";
    UlimitName["RSS"] = "rss";
    UlimitName["RTPRIO"] = "rtprio";
    UlimitName["RTTIME"] = "rttime";
    UlimitName["SIGPENDING"] = "sigpending";
    UlimitName["STACK"] = "stack";
})(UlimitName = exports.UlimitName || (exports.UlimitName = {}));
function renderUlimit(ulimit) {
    return {
        name: ulimit.name,
        softLimit: ulimit.softLimit,
        hardLimit: ulimit.hardLimit,
    };
}
var ContainerDependencyCondition;
(function (ContainerDependencyCondition) {
    /**
     * This condition emulates the behavior of links and volumes today.
     * It validates that a dependent container is started before permitting other containers to start.
     */
    ContainerDependencyCondition["START"] = "START";
    /**
     * This condition validates that a dependent container runs to completion (exits) before permitting other containers to start.
     * This can be useful for nonessential containers that run a script and then exit.
     */
    ContainerDependencyCondition["COMPLETE"] = "COMPLETE";
    /**
     * This condition is the same as COMPLETE, but it also requires that the container exits with a zero status.
     */
    ContainerDependencyCondition["SUCCESS"] = "SUCCESS";
    /**
     * This condition validates that the dependent container passes its Docker health check before permitting other containers to start.
     * This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.
     */
    ContainerDependencyCondition["HEALTHY"] = "HEALTHY";
})(ContainerDependencyCondition = exports.ContainerDependencyCondition || (exports.ContainerDependencyCondition = {}));
function renderContainerDependency(containerDependency) {
    return {
        containerName: containerDependency.container.containerName,
        condition: containerDependency.condition || ContainerDependencyCondition.HEALTHY,
    };
}
/**
 * Network protocol
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP
     */
    Protocol["TCP"] = "tcp";
    /**
     * UDP
     */
    Protocol["UDP"] = "udp";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
function renderPortMapping(pm) {
    return {
        containerPort: pm.containerPort,
        hostPort: pm.hostPort,
        protocol: pm.protocol || Protocol.TCP,
    };
}
function renderMountPoint(mp) {
    return {
        containerPath: mp.containerPath,
        readOnly: mp.readOnly,
        sourceVolume: mp.sourceVolume,
    };
}
function renderVolumeFrom(vf) {
    return {
        sourceContainer: vf.sourceContainer,
        readOnly: vf.readOnly,
    };
}
//# sourceMappingURL=data:application/json;base64,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