"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const application_multiple_target_groups_service_base_1 = require("../base/application-multiple-target-groups-service-base");
/**
 * An EC2 service running on an ECS cluster fronted by an application load balancer.
 */
class ApplicationMultipleTargetGroupsEc2Service extends application_multiple_target_groups_service_base_1.ApplicationMultipleTargetGroupsServiceBase {
    /**
     * Constructs a new instance of the ApplicationMultipleTargetGroupsEc2Service class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify only one of TaskDefinition or TaskImageOptions.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'TaskDef', {
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
            });
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                cpu: props.cpu,
                memoryLimitMiB: props.memoryLimitMiB,
                memoryReservationMiB: props.memoryReservationMiB,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
                logging: this.logDriver,
            });
            if (taskImageOptions.containerPorts) {
                for (const containerPort of taskImageOptions.containerPorts) {
                    container.addPortMappings({
                        containerPort,
                    });
                }
            }
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('At least one essential container must be specified');
        }
        if (this.taskDefinition.defaultContainer.portMappings.length === 0) {
            this.taskDefinition.defaultContainer.addPortMappings({
                containerPort: 80,
            });
        }
        this.service = this.createEc2Service(props);
        if (props.targetGroups) {
            this.addPortMappingForTargets(this.taskDefinition.defaultContainer, props.targetGroups);
            this.targetGroup = this.registerECSTargets(this.service, this.taskDefinition.defaultContainer, props.targetGroups);
        }
        else {
            this.targetGroup = this.listener.addTargets('ECS', {
                targets: [this.service],
                port: this.taskDefinition.defaultContainer.portMappings[0].containerPort,
            });
        }
    }
    createEc2Service(props) {
        return new aws_ecs_1.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: false,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
        });
    }
}
exports.ApplicationMultipleTargetGroupsEc2Service = ApplicationMultipleTargetGroupsEc2Service;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbXVsdGlwbGUtdGFyZ2V0LWdyb3Vwcy1lY3Mtc2VydmljZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFwcGxpY2F0aW9uLW11bHRpcGxlLXRhcmdldC1ncm91cHMtZWNzLXNlcnZpY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSw4Q0FBaUUsQ0FBQyxtREFBbUQ7QUFHckgsNkhBQXNLO0FBaUR0Szs7R0FFRztBQUNILE1BQWEseUNBQTBDLFNBQVEsNEZBQTBDO0lBYXJHOztPQUVHO0lBQ0gsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUF3RCxFQUFFO1FBQ2hHLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3hCLElBQUksS0FBSyxDQUFDLGNBQWMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDO1NBQ3ZGO2FBQ0ksSUFBSSxLQUFLLENBQUMsY0FBYyxFQUFFO1lBQzNCLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztTQUM5QzthQUNJLElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQzdCLE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO1lBQ2hELElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSwyQkFBaUIsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO2dCQUN6RCxhQUFhLEVBQUUsZ0JBQWdCLENBQUMsYUFBYTtnQkFDN0MsUUFBUSxFQUFFLGdCQUFnQixDQUFDLFFBQVE7YUFDdEMsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxhQUFhLEdBQUcsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7WUFDNUcsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsYUFBYSxFQUFFO2dCQUM5RCxLQUFLLEVBQUUsZ0JBQWdCLENBQUMsS0FBSztnQkFDN0IsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO2dCQUNkLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztnQkFDcEMsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLG9CQUFvQjtnQkFDaEQsV0FBVyxFQUFFLGdCQUFnQixDQUFDLFdBQVc7Z0JBQ3pDLE9BQU8sRUFBRSxnQkFBZ0IsQ0FBQyxPQUFPO2dCQUNqQyxPQUFPLEVBQUUsSUFBSSxDQUFDLFNBQVM7YUFDMUIsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxnQkFBZ0IsQ0FBQyxjQUFjLEVBQUU7Z0JBQ2pDLEtBQUssTUFBTSxhQUFhLElBQUksZ0JBQWdCLENBQUMsY0FBYyxFQUFFO29CQUN6RCxTQUFTLENBQUMsZUFBZSxDQUFDO3dCQUN0QixhQUFhO3FCQUNoQixDQUFDLENBQUM7aUJBQ047YUFDSjtTQUNKO2FBQ0k7WUFDRCxNQUFNLElBQUksS0FBSyxDQUFDLGtEQUFrRCxDQUFDLENBQUM7U0FDdkU7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRTtZQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLG9EQUFvRCxDQUFDLENBQUM7U0FDekU7UUFDRCxJQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDaEUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxlQUFlLENBQUM7Z0JBQ2pELGFBQWEsRUFBRSxFQUFFO2FBQ3BCLENBQUMsQ0FBQztTQUNOO1FBQ0QsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDNUMsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3BCLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUN4RixJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO1NBQ3RIO2FBQ0k7WUFDRCxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRTtnQkFDL0MsT0FBTyxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQztnQkFDdkIsSUFBSSxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLGFBQWE7YUFDM0UsQ0FBQyxDQUFDO1NBQ047SUFDTCxDQUFDO0lBQ08sZ0JBQWdCLENBQUMsS0FBcUQ7UUFDMUUsT0FBTyxJQUFJLG9CQUFVLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUNuQyxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQy9CLGNBQWMsRUFBRSxJQUFJLENBQUMsY0FBYztZQUNuQyxjQUFjLEVBQUUsS0FBSztZQUNyQixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtZQUNwRCxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7WUFDbEMsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLG9CQUFvQjtZQUNoRCxlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWU7U0FDekMsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztDQUNKO0FBcEZELDhGQW9GQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEVjMlNlcnZpY2UsIEVjMlRhc2tEZWZpbml0aW9uIH0gZnJvbSBcIi4uLy4uLy4uL2F3cy1lY3NcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3MnXG5pbXBvcnQgeyBBcHBsaWNhdGlvblRhcmdldEdyb3VwIH0gZnJvbSBcIi4uLy4uLy4uL2F3cy1lbGFzdGljbG9hZGJhbGFuY2luZ3YyXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2MidcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gXCIuLi8uLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQXBwbGljYXRpb25NdWx0aXBsZVRhcmdldEdyb3Vwc1NlcnZpY2VCYXNlLCBBcHBsaWNhdGlvbk11bHRpcGxlVGFyZ2V0R3JvdXBzU2VydmljZUJhc2VQcm9wcyB9IGZyb20gJy4uL2Jhc2UvYXBwbGljYXRpb24tbXVsdGlwbGUtdGFyZ2V0LWdyb3Vwcy1zZXJ2aWNlLWJhc2UnO1xuLyoqXG4gKiBUaGUgcHJvcGVydGllcyBmb3IgdGhlIEFwcGxpY2F0aW9uTXVsdGlwbGVUYXJnZXRHcm91cHNFYzJTZXJ2aWNlIHNlcnZpY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBwbGljYXRpb25NdWx0aXBsZVRhcmdldEdyb3Vwc0VjMlNlcnZpY2VQcm9wcyBleHRlbmRzIEFwcGxpY2F0aW9uTXVsdGlwbGVUYXJnZXRHcm91cHNTZXJ2aWNlQmFzZVByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgdGFzayBkZWZpbml0aW9uIHRvIHVzZSBmb3IgdGFza3MgaW4gdGhlIHNlcnZpY2UuIE9ubHkgb25lIG9mIFRhc2tEZWZpbml0aW9uIG9yIFRhc2tJbWFnZU9wdGlvbnMgbXVzdCBiZSBzcGVjaWZpZWQuXG4gICAgICpcbiAgICAgKiBbZGlzYWJsZS1hd3NsaW50OnJlZi12aWEtaW50ZXJmYWNlXVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBub25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGFza0RlZmluaXRpb24/OiBFYzJUYXNrRGVmaW5pdGlvbjtcbiAgICAvKipcbiAgICAgKiBUaGUgbWluaW11bSBudW1iZXIgb2YgQ1BVIHVuaXRzIHRvIHJlc2VydmUgZm9yIHRoZSBjb250YWluZXIuXG4gICAgICpcbiAgICAgKiBWYWxpZCB2YWx1ZXMsIHdoaWNoIGRldGVybWluZXMgeW91ciByYW5nZSBvZiB2YWxpZCB2YWx1ZXMgZm9yIHRoZSBtZW1vcnkgcGFyYW1ldGVyOlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBtaW5pbXVtIENQVSB1bml0cyByZXNlcnZlZC5cbiAgICAgKi9cbiAgICByZWFkb25seSBjcHU/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIGFtb3VudCAoaW4gTWlCKSBvZiBtZW1vcnkgdG8gcHJlc2VudCB0byB0aGUgY29udGFpbmVyLlxuICAgICAqXG4gICAgICogSWYgeW91ciBjb250YWluZXIgYXR0ZW1wdHMgdG8gZXhjZWVkIHRoZSBhbGxvY2F0ZWQgbWVtb3J5LCB0aGUgY29udGFpbmVyXG4gICAgICogaXMgdGVybWluYXRlZC5cbiAgICAgKlxuICAgICAqIEF0IGxlYXN0IG9uZSBvZiBtZW1vcnlMaW1pdE1pQiBhbmQgbWVtb3J5UmVzZXJ2YXRpb25NaUIgaXMgcmVxdWlyZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIG1lbW9yeSBsaW1pdC5cbiAgICAgKi9cbiAgICByZWFkb25seSBtZW1vcnlMaW1pdE1pQj86IG51bWJlcjtcbiAgICAvKipcbiAgICAgKiBUaGUgc29mdCBsaW1pdCAoaW4gTWlCKSBvZiBtZW1vcnkgdG8gcmVzZXJ2ZSBmb3IgdGhlIGNvbnRhaW5lci5cbiAgICAgKlxuICAgICAqIFdoZW4gc3lzdGVtIG1lbW9yeSBpcyB1bmRlciBoZWF2eSBjb250ZW50aW9uLCBEb2NrZXIgYXR0ZW1wdHMgdG8ga2VlcCB0aGVcbiAgICAgKiBjb250YWluZXIgbWVtb3J5IHRvIHRoaXMgc29mdCBsaW1pdC4gSG93ZXZlciwgeW91ciBjb250YWluZXIgY2FuIGNvbnN1bWUgbW9yZVxuICAgICAqIG1lbW9yeSB3aGVuIGl0IG5lZWRzIHRvLCB1cCB0byBlaXRoZXIgdGhlIGhhcmQgbGltaXQgc3BlY2lmaWVkIHdpdGggdGhlIG1lbW9yeVxuICAgICAqIHBhcmFtZXRlciAoaWYgYXBwbGljYWJsZSksIG9yIGFsbCBvZiB0aGUgYXZhaWxhYmxlIG1lbW9yeSBvbiB0aGUgY29udGFpbmVyXG4gICAgICogaW5zdGFuY2UsIHdoaWNoZXZlciBjb21lcyBmaXJzdC5cbiAgICAgKlxuICAgICAqIEF0IGxlYXN0IG9uZSBvZiBtZW1vcnlMaW1pdE1pQiBhbmQgbWVtb3J5UmVzZXJ2YXRpb25NaUIgaXMgcmVxdWlyZWQuXG4gICAgICpcbiAgICAgKiBOb3RlIHRoYXQgdGhpcyBzZXR0aW5nIHdpbGwgYmUgaWdub3JlZCBpZiBUYXNrSW1hZ2VzT3B0aW9ucyBpcyBzcGVjaWZpZWRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gbWVtb3J5IHJlc2VydmVkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1lbW9yeVJlc2VydmF0aW9uTWlCPzogbnVtYmVyO1xufVxuLyoqXG4gKiBBbiBFQzIgc2VydmljZSBydW5uaW5nIG9uIGFuIEVDUyBjbHVzdGVyIGZyb250ZWQgYnkgYW4gYXBwbGljYXRpb24gbG9hZCBiYWxhbmNlci5cbiAqL1xuZXhwb3J0IGNsYXNzIEFwcGxpY2F0aW9uTXVsdGlwbGVUYXJnZXRHcm91cHNFYzJTZXJ2aWNlIGV4dGVuZHMgQXBwbGljYXRpb25NdWx0aXBsZVRhcmdldEdyb3Vwc1NlcnZpY2VCYXNlIHtcbiAgICAvKipcbiAgICAgKiBUaGUgRUMyIHNlcnZpY2UgaW4gdGhpcyBjb25zdHJ1Y3QuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IHNlcnZpY2U6IEVjMlNlcnZpY2U7XG4gICAgLyoqXG4gICAgICogVGhlIEVDMiBUYXNrIERlZmluaXRpb24gaW4gdGhpcyBjb25zdHJ1Y3QuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IHRhc2tEZWZpbml0aW9uOiBFYzJUYXNrRGVmaW5pdGlvbjtcbiAgICAvKipcbiAgICAgKiBUaGUgZGVmYXVsdCB0YXJnZXQgZ3JvdXAgZm9yIHRoZSBzZXJ2aWNlLlxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSB0YXJnZXRHcm91cDogQXBwbGljYXRpb25UYXJnZXRHcm91cDtcbiAgICAvKipcbiAgICAgKiBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBBcHBsaWNhdGlvbk11bHRpcGxlVGFyZ2V0R3JvdXBzRWMyU2VydmljZSBjbGFzcy5cbiAgICAgKi9cbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXBwbGljYXRpb25NdWx0aXBsZVRhcmdldEdyb3Vwc0VjMlNlcnZpY2VQcm9wcyA9IHt9KSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgICAgICBpZiAocHJvcHMudGFza0RlZmluaXRpb24gJiYgcHJvcHMudGFza0ltYWdlT3B0aW9ucykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgbXVzdCBzcGVjaWZ5IG9ubHkgb25lIG9mIFRhc2tEZWZpbml0aW9uIG9yIFRhc2tJbWFnZU9wdGlvbnMuJyk7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSBpZiAocHJvcHMudGFza0RlZmluaXRpb24pIHtcbiAgICAgICAgICAgIHRoaXMudGFza0RlZmluaXRpb24gPSBwcm9wcy50YXNrRGVmaW5pdGlvbjtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIGlmIChwcm9wcy50YXNrSW1hZ2VPcHRpb25zKSB7XG4gICAgICAgICAgICBjb25zdCB0YXNrSW1hZ2VPcHRpb25zID0gcHJvcHMudGFza0ltYWdlT3B0aW9ucztcbiAgICAgICAgICAgIHRoaXMudGFza0RlZmluaXRpb24gPSBuZXcgRWMyVGFza0RlZmluaXRpb24odGhpcywgJ1Rhc2tEZWYnLCB7XG4gICAgICAgICAgICAgICAgZXhlY3V0aW9uUm9sZTogdGFza0ltYWdlT3B0aW9ucy5leGVjdXRpb25Sb2xlLFxuICAgICAgICAgICAgICAgIHRhc2tSb2xlOiB0YXNrSW1hZ2VPcHRpb25zLnRhc2tSb2xlLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICBjb25zdCBjb250YWluZXJOYW1lID0gdGFza0ltYWdlT3B0aW9ucy5jb250YWluZXJOYW1lICE9PSB1bmRlZmluZWQgPyB0YXNrSW1hZ2VPcHRpb25zLmNvbnRhaW5lck5hbWUgOiAnd2ViJztcbiAgICAgICAgICAgIGNvbnN0IGNvbnRhaW5lciA9IHRoaXMudGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKGNvbnRhaW5lck5hbWUsIHtcbiAgICAgICAgICAgICAgICBpbWFnZTogdGFza0ltYWdlT3B0aW9ucy5pbWFnZSxcbiAgICAgICAgICAgICAgICBjcHU6IHByb3BzLmNwdSxcbiAgICAgICAgICAgICAgICBtZW1vcnlMaW1pdE1pQjogcHJvcHMubWVtb3J5TGltaXRNaUIsXG4gICAgICAgICAgICAgICAgbWVtb3J5UmVzZXJ2YXRpb25NaUI6IHByb3BzLm1lbW9yeVJlc2VydmF0aW9uTWlCLFxuICAgICAgICAgICAgICAgIGVudmlyb25tZW50OiB0YXNrSW1hZ2VPcHRpb25zLmVudmlyb25tZW50LFxuICAgICAgICAgICAgICAgIHNlY3JldHM6IHRhc2tJbWFnZU9wdGlvbnMuc2VjcmV0cyxcbiAgICAgICAgICAgICAgICBsb2dnaW5nOiB0aGlzLmxvZ0RyaXZlcixcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgaWYgKHRhc2tJbWFnZU9wdGlvbnMuY29udGFpbmVyUG9ydHMpIHtcbiAgICAgICAgICAgICAgICBmb3IgKGNvbnN0IGNvbnRhaW5lclBvcnQgb2YgdGFza0ltYWdlT3B0aW9ucy5jb250YWluZXJQb3J0cykge1xuICAgICAgICAgICAgICAgICAgICBjb250YWluZXIuYWRkUG9ydE1hcHBpbmdzKHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGNvbnRhaW5lclBvcnQsXG4gICAgICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignWW91IG11c3Qgc3BlY2lmeSBvbmUgb2Y6IHRhc2tEZWZpbml0aW9uIG9yIGltYWdlJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKCF0aGlzLnRhc2tEZWZpbml0aW9uLmRlZmF1bHRDb250YWluZXIpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIGVzc2VudGlhbCBjb250YWluZXIgbXVzdCBiZSBzcGVjaWZpZWQnKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAodGhpcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyLnBvcnRNYXBwaW5ncy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIHRoaXMudGFza0RlZmluaXRpb24uZGVmYXVsdENvbnRhaW5lci5hZGRQb3J0TWFwcGluZ3Moe1xuICAgICAgICAgICAgICAgIGNvbnRhaW5lclBvcnQ6IDgwLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5zZXJ2aWNlID0gdGhpcy5jcmVhdGVFYzJTZXJ2aWNlKHByb3BzKTtcbiAgICAgICAgaWYgKHByb3BzLnRhcmdldEdyb3Vwcykge1xuICAgICAgICAgICAgdGhpcy5hZGRQb3J0TWFwcGluZ0ZvclRhcmdldHModGhpcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyLCBwcm9wcy50YXJnZXRHcm91cHMpO1xuICAgICAgICAgICAgdGhpcy50YXJnZXRHcm91cCA9IHRoaXMucmVnaXN0ZXJFQ1NUYXJnZXRzKHRoaXMuc2VydmljZSwgdGhpcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyLCBwcm9wcy50YXJnZXRHcm91cHMpO1xuICAgICAgICB9XG4gICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgdGhpcy50YXJnZXRHcm91cCA9IHRoaXMubGlzdGVuZXIuYWRkVGFyZ2V0cygnRUNTJywge1xuICAgICAgICAgICAgICAgIHRhcmdldHM6IFt0aGlzLnNlcnZpY2VdLFxuICAgICAgICAgICAgICAgIHBvcnQ6IHRoaXMudGFza0RlZmluaXRpb24uZGVmYXVsdENvbnRhaW5lci5wb3J0TWFwcGluZ3NbMF0uY29udGFpbmVyUG9ydCxcbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgfVxuICAgIHByaXZhdGUgY3JlYXRlRWMyU2VydmljZShwcm9wczogQXBwbGljYXRpb25NdWx0aXBsZVRhcmdldEdyb3Vwc0VjMlNlcnZpY2VQcm9wcyk6IEVjMlNlcnZpY2Uge1xuICAgICAgICByZXR1cm4gbmV3IEVjMlNlcnZpY2UodGhpcywgJ1NlcnZpY2UnLCB7XG4gICAgICAgICAgICBjbHVzdGVyOiB0aGlzLmNsdXN0ZXIsXG4gICAgICAgICAgICBkZXNpcmVkQ291bnQ6IHRoaXMuZGVzaXJlZENvdW50LFxuICAgICAgICAgICAgdGFza0RlZmluaXRpb246IHRoaXMudGFza0RlZmluaXRpb24sXG4gICAgICAgICAgICBhc3NpZ25QdWJsaWNJcDogZmFsc2UsXG4gICAgICAgICAgICBzZXJ2aWNlTmFtZTogcHJvcHMuc2VydmljZU5hbWUsXG4gICAgICAgICAgICBoZWFsdGhDaGVja0dyYWNlUGVyaW9kOiBwcm9wcy5oZWFsdGhDaGVja0dyYWNlUGVyaW9kLFxuICAgICAgICAgICAgcHJvcGFnYXRlVGFnczogcHJvcHMucHJvcGFnYXRlVGFncyxcbiAgICAgICAgICAgIGVuYWJsZUVDU01hbmFnZWRUYWdzOiBwcm9wcy5lbmFibGVFQ1NNYW5hZ2VkVGFncyxcbiAgICAgICAgICAgIGNsb3VkTWFwT3B0aW9uczogcHJvcHMuY2xvdWRNYXBPcHRpb25zLFxuICAgICAgICB9KTtcbiAgICB9XG59XG4iXX0=