"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assets = require("../../assets"); // Automatically re-written from '@aws-cdk/assets'
const ecr = require("../../aws-ecr"); // Automatically re-written from '@aws-cdk/aws-ecr'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const fs = require("fs");
const minimatch = require("minimatch");
const path = require("path");
/**
 * An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 */
class DockerImageAsset extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        const file = path.join(dir, props.file || 'Dockerfile');
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            exclude = [...exclude, ...fs.readFileSync(ignore).toString().split('\n').filter(e => !!e)];
        }
        // make sure the docker file and the dockerignore file end up in the staging area
        // see https://github.com/aws/aws-cdk/issues/6004
        exclude = exclude.filter(ignoreExpression => {
            return !(minimatch(file, ignoreExpression, { matchBase: true }) ||
                minimatch(ignore, ignoreExpression, { matchBase: true }));
        });
        if (props.repositoryName) {
            this.node.addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (props.target) {
            extraHash.target = props.target;
        }
        if (props.file) {
            extraHash.file = props.file;
        }
        if (props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new assets.Staging(this, 'Staging', {
            ...props,
            exclude,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.sourceHash;
        const stack = core_1.Stack.of(this);
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: staging.stagedPath,
            dockerBuildArgs: props.buildArgs,
            dockerBuildTarget: props.target,
            dockerFile: props.file,
            repositoryName: props.repositoryName,
            sourceHash: staging.sourceHash,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
}
exports.DockerImageAsset = DockerImageAsset;
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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