"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
class VpcEndpoint extends core_1.Resource {
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement
     * must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add
     */
    addToPolicy(statement) {
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
/**
 * The type of VPC endpoint.
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    /**
     * Interface
     *
     * An interface endpoint is an elastic network interface with a private IP
     * address that serves as an entry point for traffic destined to a supported
     * service.
     */
    VpcEndpointType["INTERFACE"] = "Interface";
    /**
     * Gateway
     *
     * A gateway endpoint is a gateway that is a target for a specified route in
     * your route table, used for traffic destined to a supported AWS service.
     */
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * An AWS service for a gateway VPC endpoint.
 */
class GatewayVpcEndpointAwsService {
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * A gateway VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        super(scope, id);
        const subnets = props.subnets
            ? util_1.flatten(props.subnets.map(s => props.vpc.selectSubnets(s).subnets))
            : [...props.vpc.privateSubnets, ...props.vpc.publicSubnets, ...props.vpc.isolatedSubnets];
        const routeTableIds = util_1.allRouteTableIds(subnets);
        if (routeTableIds.length === 0) {
            throw new Error('Can\'t add a gateway endpoint to VPC; route table IDs are not available');
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
/**
 * A custom-hosted service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointService {
    constructor(name, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = false;
        this.name = name;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointService = InterfaceVpcEndpointService;
/**
 * An AWS service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointAwsService {
    constructor(name, prefix, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = true;
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuild-fips');
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM = new InterfaceVpcEndpointAwsService('elasticfilesystem');
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM_FIPS = new InterfaceVpcEndpointAwsService('elasticfilesystem-fips');
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
InterfaceVpcEndpointAwsService.STORAGE_GATEWAY = new InterfaceVpcEndpointAwsService('storagegateway');
InterfaceVpcEndpointAwsService.REKOGNITION = new InterfaceVpcEndpointAwsService('rekognition');
InterfaceVpcEndpointAwsService.REKOGNITION_FIPS = new InterfaceVpcEndpointAwsService('rekognition-fips');
/**
 * A interface VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        const securityGroups = props.securityGroups || [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.securityGroupId = securityGroups[0].securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups,
        });
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(peer_1.Peer.ipv4(props.vpc.vpcCidrBlock));
        }
        const lookupSupportedAzs = (_a = props.lookupSupportedAzs) !== null && _a !== void 0 ? _a : false;
        const subnetSelection = props.vpc.selectSubnets({ ...props.subnets, onePerAz: true });
        let subnets;
        // If we don't have an account/region, we will not be able to do filtering on AZs since
        // they will be undefined
        // Otherwise, we filter by AZ
        const agnostic = (core_1.Token.isUnresolved(this.stack.account) || core_1.Token.isUnresolved(this.stack.region));
        if (agnostic && lookupSupportedAzs) {
            throw new Error('Cannot look up VPC endpoint availability zones if account/region are not specified');
        }
        else if (!agnostic && lookupSupportedAzs) {
            const availableAZs = this.availableAvailabilityZones(props.service.name);
            subnets = subnetSelection.subnets.filter(s => availableAZs.includes(s.availabilityZone));
        }
        else {
            subnets = subnetSelection.subnets;
        }
        const subnetIds = subnets.map(s => s.subnetId);
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: (_c = (_b = props.privateDnsEnabled) !== null && _b !== void 0 ? _b : props.service.privateDnsDefault) !== null && _c !== void 0 ? _c : true,
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * Imports an existing interface VPC endpoint.
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        if (!attrs.securityGroups && !attrs.securityGroupId) {
            throw new Error('Either `securityGroups` or `securityGroupId` must be specified.');
        }
        const securityGroups = attrs.securityGroupId
            ? [security_group_1.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)]
            : attrs.securityGroups;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups,
                });
            }
        }
        return new Import(scope, id);
    }
    availableAvailabilityZones(serviceName) {
        // Here we check what AZs the endpoint service is available in
        // If for whatever reason we can't retrieve the AZs, and no context is set,
        // we will fall back to all AZs
        const availableAZs = core_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.ENDPOINT_SERVICE_AVAILABILITY_ZONE_PROVIDER,
            dummyValue: this.stack.availabilityZones,
            props: { serviceName },
        }).value;
        if (!Array.isArray(availableAZs)) {
            throw new Error(`Discovered AZs for endpoint service ${serviceName} must be an array`);
        }
        return availableAZs;
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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