"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * InvalidCidrRangeError is thrown when attempting to perform operations on a CIDR
 * range that is either not valid, or outside of the VPC size limits.
 */
class InvalidCidrRangeError extends Error {
    constructor(cidr) {
        super(cidr + ' is not a valid VPC CIDR range (must be between /16 and /28)');
        // The following line is required for type checking of custom errors, and must be called right after super()
        // https://stackoverflow.com/questions/31626231/custom-error-class-in-typescript
        Object.setPrototypeOf(this, InvalidCidrRangeError.prototype);
    }
}
exports.InvalidCidrRangeError = InvalidCidrRangeError;
/**
 * NetworkUtils contains helpers to work with network constructs (subnets/ranges)
 */
class NetworkUtils {
    /**
     * Validates an IPv4 string
     *
     * returns true of the string contains 4 numbers between 0-255 delimited by
     * a `.` character
     */
    static validIp(ipAddress) {
        const octets = ipAddress.split('.');
        if (octets.length !== 4) {
            return false;
        }
        return octets.map((octet) => parseInt(octet, 10)).
            filter((octet) => octet >= 0 && octet <= 255).length === 4;
    }
    /**
     * Converts a string IPv4 to a number
     *
     * takes an IP Address (e.g. 174.66.173.168) and converts to a number
     * (e.g 2923605416); currently only supports IPv4
     *
     * Uses the formula:
     * (first octet * 256³) + (second octet * 256²) + (third octet * 256) +
     * (fourth octet)
     *
     * @param  {string} the IP address (e.g. 174.66.173.168)
     * @returns {number} the integer value of the IP address (e.g 2923605416)
     */
    static ipToNum(ipAddress) {
        if (!this.validIp(ipAddress)) {
            throw new Error(`${ipAddress} is not valid`);
        }
        return ipAddress
            .split('.')
            .reduce((p, c, i) => p + parseInt(c, 10) * 256 ** (3 - i), 0);
    }
    /**
     * Takes number and converts it to IPv4 address string
     *
     * Takes a number (e.g 2923605416) and converts it to an IPv4 address string
     * currently only supports IPv4
     *
     * @param  {number} the integer value of the IP address (e.g 2923605416)
     * @returns {string} the IPv4 address (e.g. 174.66.173.168)
     */
    static numToIp(ipNum) {
        // this all because bitwise math is signed
        let remaining = ipNum;
        const address = new Array();
        for (let i = 0; i < 4; i++) {
            if (remaining !== 0) {
                address.push(Math.floor(remaining / 256 ** (3 - i)));
                remaining = remaining % 256 ** (3 - i);
            }
            else {
                address.push(0);
            }
        }
        const ipAddress = address.join('.');
        if (!this.validIp(ipAddress)) {
            throw new Error(`${ipAddress} is not a valid IP Address`);
        }
        return ipAddress;
    }
}
exports.NetworkUtils = NetworkUtils;
/**
 * Creates a network based on a CIDR Block to build contained subnets
 */
class NetworkBuilder {
    /**
     * Create a network using the provided CIDR block
     *
     * No subnets are allocated in the constructor, the maxIpConsumed is set one
     * less than the first IP in the network
     *
     */
    constructor(cidr) {
        /**
         * The list of CIDR blocks for subnets within this network
         */
        this.subnetCidrs = [];
        this.networkCidr = new CidrBlock(cidr);
        this.subnetCidrs = [];
        this.nextAvailableIp = this.networkCidr.minAddress();
    }
    /**
     * Add a subnet to the network and update the maxIpConsumed
     */
    addSubnet(mask) {
        return this.addSubnets(mask, 1)[0];
    }
    /**
     * Add {count} number of subnets to the network and update the maxIpConsumed
     */
    addSubnets(mask, count = 1) {
        if (mask < 16 || mask > 28) {
            throw new InvalidCidrRangeError(`x.x.x.x/${mask}`);
        }
        const maxIp = this.nextAvailableIp + (CidrBlock.calculateNetsize(mask) * count);
        if (this.networkCidr.maxAddress() < maxIp - 1) {
            throw new Error(`${count} of /${mask} exceeds remaining space of ${this.networkCidr.cidr}`);
        }
        const subnets = [];
        for (let i = 0; i < count; i++) {
            const subnet = new CidrBlock(this.nextAvailableIp, mask);
            this.nextAvailableIp = subnet.nextBlock().minAddress();
            this.subnetCidrs.push(subnet);
            subnets.push(subnet);
        }
        return subnets.map((subnet) => (subnet.cidr));
    }
    /**
     * return the CIDR notation strings for all subnets in the network
     */
    get cidrStrings() {
        return this.subnetCidrs.map((subnet) => (subnet.cidr));
    }
    /**
     * Calculates the largest subnet to create of the given count from the
     * remaining IP space
     */
    maskForRemainingSubnets(subnetCount) {
        const remaining = this.networkCidr.maxAddress() - this.nextAvailableIp + 1;
        const ipsPerSubnet = Math.floor(remaining / subnetCount);
        return 32 - Math.floor(Math.log2(ipsPerSubnet));
    }
}
exports.NetworkBuilder = NetworkBuilder;
/**
 * A block of IP address space with a given bit prefix
 */
class CidrBlock {
    constructor(ipAddressOrCidr, mask) {
        if (typeof ipAddressOrCidr === 'string') {
            this.mask = parseInt(ipAddressOrCidr.split('/')[1], 10);
            this.networkAddress = NetworkUtils.ipToNum(ipAddressOrCidr.split('/')[0]) +
                CidrBlock.calculateNetsize(this.mask) - 1;
        }
        else {
            if (typeof mask === 'number') {
                this.mask = mask;
            }
            else {
                // this should be impossible
                this.mask = 16;
            }
            this.networkAddress = ipAddressOrCidr + CidrBlock.calculateNetsize(this.mask) - 1;
            this.networkSize = 2 ** (32 - this.mask);
        }
        this.networkSize = 2 ** (32 - this.mask);
        this.cidr = `${this.minIp()}/${this.mask}`;
    }
    /**
     * Calculates the netmask for a given CIDR mask
     *
     * For example:
     * CidrBlock.calculateNetmask(24) returns '255.255.255.0'
     */
    static calculateNetmask(mask) {
        return NetworkUtils.numToIp(2 ** 32 - 2 ** (32 - mask));
    }
    /**
     * Calculates the number IP addresses in a CIDR Mask
     *
     * For example:
     * CidrBlock.calculateNetsize(24) returns 256
     */
    static calculateNetsize(mask) {
        return 2 ** (32 - mask);
    }
    /*
     * The maximum IP in the CIDR Blcok e.g. '10.0.8.255'
     */
    maxIp() {
        // min + (2^(32-mask)) - 1 [zero needs to count]
        return NetworkUtils.numToIp(this.maxAddress());
    }
    /*
     * The minimum IP in the CIDR Blcok e.g. '10.0.0.0'
     */
    minIp() {
        return NetworkUtils.numToIp(this.minAddress());
    }
    /*
     * Returns the number representation for the minimum IPv4 address
     */
    minAddress() {
        const div = this.networkAddress % this.networkSize;
        return this.networkAddress - div;
    }
    /*
     * Returns the number representation for the maximum IPv4 address
     */
    maxAddress() {
        // min + (2^(32-mask)) - 1 [zero needs to count]
        return this.minAddress() + this.networkSize - 1;
    }
    /*
     * Returns the next CIDR Block of the same mask size
     */
    nextBlock() {
        return new CidrBlock(this.maxAddress() + 1, this.mask);
    }
    /*
     * Returns true if this CidrBlock fully contains the provided CidrBlock
     */
    containsCidr(other) {
        return (this.maxAddress() >= other.maxAddress()) &&
            (this.minAddress() <= other.minAddress());
    }
}
exports.CidrBlock = CidrBlock;
//# sourceMappingURL=data:application/json;base64,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