"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Either an IPv4 or an IPv6 CIDR
 *
 * @experimental
 */
class AclCidr {
    /**
     * An IP network range in CIDR notation (for example, 172.16.0.0/24).
     */
    static ipv4(ipv4Cidr) {
        return new AclCidrImpl({
            cidrBlock: ipv4Cidr,
        });
    }
    /**
     * The CIDR containing all IPv4 addresses (i.e., 0.0.0.0/0)
     */
    static anyIpv4() {
        return AclCidr.ipv4('0.0.0.0/0');
    }
    /**
     * An IPv6 network range in CIDR notation (for example, 2001:db8::/48)
     */
    static ipv6(ipv6Cidr) {
        return new AclCidrImpl({
            ipv6CidrBlock: ipv6Cidr,
        });
    }
    /**
     * The CIDR containing all IPv6 addresses (i.e., ::/0)
     */
    static anyIpv6() {
        return AclCidr.ipv6('::/0');
    }
}
exports.AclCidr = AclCidr;
class AclCidrImpl extends AclCidr {
    constructor(config) {
        super();
        this.config = config;
    }
    toCidrConfig() {
        return this.config;
    }
}
/**
 * The traffic that is configured using a Network ACL entry
 *
 * @experimental
 */
class AclTraffic {
    /**
     * Apply the ACL entry to all traffic
     */
    static allTraffic() {
        return new AclTrafficImpl({
            protocol: -1,
        });
    }
    /**
     * Apply the ACL entry to ICMP traffic of given type and code
     */
    static icmp(props) {
        return new AclTrafficImpl({
            protocol: 1,
            icmp: props,
        });
    }
    /**
     * Apply the ACL entry to ICMPv6 traffic of given type and code
     *
     * Requires an IPv6 CIDR block.
     */
    static icmpv6(props) {
        return new AclTrafficImpl({
            protocol: 58,
            icmp: props,
        });
    }
    /**
     * Apply the ACL entry to TCP traffic on a given port
     */
    static tcpPort(port) {
        return new AclTrafficImpl({
            protocol: 6,
            portRange: {
                from: port,
                to: port,
            },
        });
    }
    /**
     * Apply the ACL entry to TCP traffic on a given port range
     */
    static tcpPortRange(startPort, endPort) {
        return new AclTrafficImpl({
            protocol: 6,
            portRange: {
                from: startPort,
                to: endPort,
            },
        });
    }
    /**
     * Apply the ACL entry to UDP traffic on a given port
     */
    static udpPort(port) {
        return new AclTrafficImpl({
            protocol: 17,
            portRange: {
                from: port,
                to: port,
            },
        });
    }
    /**
     * Apply the ACL entry to UDP traffic on a given port range
     */
    static udpPortRange(startPort, endPort) {
        return new AclTrafficImpl({
            protocol: 17,
            portRange: {
                from: startPort,
                to: endPort,
            },
        });
    }
}
exports.AclTraffic = AclTraffic;
class AclTrafficImpl extends AclTraffic {
    constructor(config) {
        super();
        this.config = config;
    }
    toTrafficConfig() {
        return this.config;
    }
}
//# sourceMappingURL=data:application/json;base64,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