"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const peer_1 = require("./peer");
/**
 * Manage the allowed network connections for constructs with Security Groups.
 *
 * Security Groups can be thought of as a firewall for network-connected
 * devices. This class makes it easy to allow network connections to and
 * from security groups, and between security groups individually. When
 * establishing connectivity between security groups, it will automatically
 * add rules in both security groups
 *
 * This object can manage one or more security groups.
 */
class Connections {
    constructor(props = {}) {
        /**
         * Underlying securityGroup for this Connections object, if present
         *
         * May be empty if this Connections object is not managing a SecurityGroup,
         * but simply representing a Connectable peer.
         */
        this._securityGroups = new ReactiveList();
        /**
         * The rule that defines how to represent this peer in a security group
         */
        this._securityGroupRules = new ReactiveList();
        /**
         * When doing bidirectional grants between Connections, make sure we don't recursive infinitely
         */
        this.skip = false;
        /**
         * When doing bidirectional grants between Security Groups in different stacks, put the rule on the other SG
         */
        this.remoteRule = false;
        this.connections = this;
        this._securityGroups.push(...(props.securityGroups || []));
        this._securityGroupRules.push(...this._securityGroups.asArray());
        if (props.peer) {
            this._securityGroupRules.push(props.peer);
        }
        this.defaultPort = props.defaultPort;
    }
    get securityGroups() {
        return this._securityGroups.asArray();
    }
    /**
     * Add a security group to the list of security groups managed by this object
     */
    addSecurityGroup(...securityGroups) {
        for (const securityGroup of securityGroups) {
            this._securityGroups.push(securityGroup);
            this._securityGroupRules.push(securityGroup);
        }
    }
    /**
     * Allow connections to the peer on the given port
     */
    allowTo(other, portRange, description) {
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addEgressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowFrom(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * Allow connections from the peer on the given port
     */
    allowFrom(other, portRange, description) {
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowTo(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * Allow hosts inside the security group to connect to each other on the given port
     */
    allowInternally(portRange, description) {
        this._securityGroups.forEachAndForever(securityGroup => {
            this._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description);
                securityGroup.addEgressRule(rule, portRange, description);
            });
        });
    }
    /**
     * Allow to all IPv4 ranges
     */
    allowToAnyIpv4(portRange, description) {
        this.allowTo(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * Allow from any IPv4 ranges
     */
    allowFromAnyIpv4(portRange, description) {
        this.allowFrom(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * Allow connections from the peer on our default port
     *
     * Even if the peer has a default port, we will always use our default port.
     */
    allowDefaultPortFrom(other, description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFrom(): this resource has no default port');
        }
        this.allowFrom(other, this.defaultPort, description);
    }
    /**
     * Allow hosts inside the security group to connect to each other
     */
    allowDefaultPortInternally(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortInternally(): this resource has no default port');
        }
        this.allowInternally(this.defaultPort, description);
    }
    /**
     * Allow default connections from all IPv4 ranges
     */
    allowDefaultPortFromAnyIpv4(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFromAnyIpv4(): this resource has no default port');
        }
        this.allowFromAnyIpv4(this.defaultPort, description);
    }
    /**
     * Allow connections to the security group on their default port
     */
    allowToDefaultPort(other, description) {
        if (other.connections.defaultPort === undefined) {
            throw new Error('Cannot call alloToDefaultPort(): other resource has no default port');
        }
        this.allowTo(other, other.connections.defaultPort, description);
    }
    /**
     * Allow connections from the peer on our default port
     *
     * Even if the peer has a default port, we will always use our default port.
     */
    allowDefaultPortTo(other, description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortTo(): this resource has no default port');
        }
        this.allowTo(other, this.defaultPort, description);
    }
}
exports.Connections = Connections;
class ReactiveList {
    constructor() {
        this.elements = new Array();
        this.listeners = new Array();
    }
    push(...xs) {
        this.elements.push(...xs);
        for (const listener of this.listeners) {
            for (const x of xs) {
                listener(x);
            }
        }
    }
    forEachAndForever(listener) {
        for (const element of this.elements) {
            listener(element);
        }
        this.listeners.push(listener);
    }
    asArray() {
        return this.elements.slice();
    }
    get length() {
        return this.elements.length;
    }
}
//# sourceMappingURL=data:application/json;base64,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