"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cognito_generated_1 = require("./cognito.generated");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.identityProviders = [];
    }
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
}
/**
 * Define a Cognito User Pool
 */
class UserPool extends UserPoolBase {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_a = props.userInvitation) === null || _a === void 0 ? void 0 : _a.emailBody,
            emailSubject: (_b = props.userInvitation) === null || _b === void 0 ? void 0 : _b.emailSubject,
            smsMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration: undefinedIfNoKeys({
                from: (_d = props.emailSettings) === null || _d === void 0 ? void 0 : _d.from,
                replyToEmailAddress: (_e = props.emailSettings) === null || _e === void 0 ? void 0 : _e.replyTo,
            }),
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends UserPoolBase {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        return UserPool.fromUserPoolId(scope, id, core_1.Stack.of(scope).parseArn(userPoolArn).resourceName);
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        this.triggers[operation.operationName] = fn.functionArn;
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn,
        });
    }
    verificationMessageConfiguration(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailStyle) !== null && _b !== void 0 ? _b : VerificationEmailStyle.CODE;
        const emailSubject = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailSubject) !== null && _d !== void 0 ? _d : 'Verify your new account';
        const smsMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage) !== null && _f !== void 0 ? _f : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailBody) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases) !== null && _a !== void 0 ? _a : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a;
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
            };
        }
        else {
            const smsRoleExternalId = this.node.uniqueId.substr(0, 1223); // sts:ExternalId max length of 1224
            const smsRole = (_a = props.smsRole) !== null && _a !== void 0 ? _a : new aws_iam_1.Role(this, 'smsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                    conditions: {
                        StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                    },
                }),
                inlinePolicies: {
                    /*
                     * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                     * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                     * Seems like a case of being excessively strict.
                     */
                    'sns-publish': new aws_iam_1.PolicyDocument({
                        statements: [
                            new aws_iam_1.PolicyStatement({
                                actions: ['sns:Publish'],
                                resources: ['*'],
                            }),
                        ],
                    }),
                },
            });
            return {
                externalId: smsRoleExternalId,
                snsCallerArn: smsRole.roleArn,
            };
        }
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _a, _b, _c, _d, _e, _f;
        const tempPasswordValidity = (_a = props.passwordPolicy) === null || _a === void 0 ? void 0 : _a.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_b = props.passwordPolicy.minLength) !== null && _b !== void 0 ? _b : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.requireLowercase,
            requireUppercase: (_d = props.passwordPolicy) === null || _d === void 0 ? void 0 : _d.requireUppercase,
            requireNumbers: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireDigits,
            requireSymbols: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => {
                var _a, _b;
                return ({
                    name: StandardAttributeNames[attrName],
                    mutable: (_a = attr.mutable) !== null && _a !== void 0 ? _a : true,
                    required: (_b = attr.required) !== null && _b !== void 0 ? _b : false,
                });
            });
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_b = (_a = attrConfig.numberConstraints) === null || _a === void 0 ? void 0 : _a.min) === null || _b === void 0 ? void 0 : _b.toString(),
                    maxValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.max) === null || _d === void 0 ? void 0 : _d.toString(),
                };
                const stringConstraints = {
                    minLength: (_f = (_e = attrConfig.stringConstraints) === null || _e === void 0 ? void 0 : _e.minLen) === null || _f === void 0 ? void 0 : _f.toString(),
                    maxLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.maxLen) === null || _h === void 0 ? void 0 : _h.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
}
exports.UserPool = UserPool;
const StandardAttributeNames = {
    address: 'address',
    birthdate: 'birthdate',
    email: 'email',
    familyName: 'family_name',
    gender: 'gender',
    givenName: 'given_name',
    locale: 'locale',
    middleName: 'middle_name',
    fullname: 'name',
    nickname: 'nickname',
    phoneNumber: 'phone_number',
    profilePicture: 'picture',
    preferredUsername: 'preferred_username',
    profilePage: 'profile',
    timezone: 'zoneinfo',
    lastUpdateTime: 'updated_at',
    website: 'website',
};
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).reduce((acc, v) => acc && (v === undefined), true);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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