"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const custom_resources_1 = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const cognito_generated_1 = require("./cognito.generated");
/**
 * Define a user pool domain
 */
class UserPoolDomain extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (((_a = props.cognitoDomain) === null || _a === void 0 ? void 0 : _a.domainPrefix) &&
            !core_1.Token.isUnresolved((_b = props.cognitoDomain) === null || _b === void 0 ? void 0 : _b.domainPrefix) &&
            !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        this.isCognitoDomain = !!props.cognitoDomain;
        const domainName = ((_c = props.cognitoDomain) === null || _c === void 0 ? void 0 : _c.domainPrefix) || ((_d = props.customDomain) === null || _d === void 0 ? void 0 : _d.domainName);
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * The domain name of the CloudFront distribution associated with the user pool domain.
     */
    get cloudFrontDomainName() {
        const sdkCall = {
            service: 'CognitoIdentityServiceProvider',
            action: 'describeUserPoolDomain',
            parameters: {
                Domain: this.domainName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
        };
        const customResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
            resourceType: 'Custom::UserPoolCloudFrontDomainName',
            onCreate: sdkCall,
            onUpdate: sdkCall,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                // DescribeUserPoolDomain only supports access level '*'
                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                resources: ['*'],
            }),
        });
        return customResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
    /**
     * The URL to the hosted UI associated with this domain
     */
    baseUrl() {
        if (this.isCognitoDomain) {
            return `https://${this.domainName}.auth.${core_1.Stack.of(this).region}.amazoncognito.com`;
        }
        return `https://${this.domainName}`;
    }
    /**
     * The URL to the sign in page in this domain using a specific UserPoolClient
     * @param client [disable-awslint:ref-via-interface] the user pool client that the UI will use to interact with the UserPool
     * @param options options to customize the behaviour of this method.
     */
    signInUrl(client, options) {
        var _a;
        let responseType;
        if (client.oAuthFlows.authorizationCodeGrant) {
            responseType = 'code';
        }
        else if (client.oAuthFlows.implicitCodeGrant) {
            responseType = 'token';
        }
        else {
            throw new Error('signInUrl is not supported for clients without authorizationCodeGrant or implicitCodeGrant flow enabled');
        }
        const path = (_a = options.signInPath) !== null && _a !== void 0 ? _a : '/login';
        return `${this.baseUrl()}${path}?client_id=${client.userPoolClientId}&response_type=${responseType}&redirect_uri=${options.redirectUri}`;
    }
}
exports.UserPoolDomain = UserPoolDomain;
//# sourceMappingURL=data:application/json;base64,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