"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("./action");
/**
 * Validation function that checks if the number of artifacts is within the given bounds
 */
function validateArtifactBounds(type, artifacts, min, max, category, provider) {
    const ret = [];
    if (artifacts.length < min) {
        ret.push(`${category}/${provider} must have at least ${min} ${type} artifacts`);
    }
    if (artifacts.length > max) {
        ret.push(`${category}/${provider} cannot have more than ${max} ${type} artifacts`);
    }
    return ret;
}
exports.validateArtifactBounds = validateArtifactBounds;
/**
 * Validation function that guarantees that an action is or is not a source action. This is useful because Source actions can only be
 * in the first stage of a pipeline, and the first stage can only contain source actions.
 */
function validateSourceAction(mustBeSource, category, actionName, stageName) {
    if (mustBeSource !== (category === action_1.ActionCategory.SOURCE)) {
        return [`Action ${actionName} in stage ${stageName}: ` + (mustBeSource ? 'first stage may only contain Source actions'
                : 'Source actions may only occur in first stage')];
    }
    return [];
}
exports.validateSourceAction = validateSourceAction;
/**
 * Regex to validate Pipeline, Stage, Action names
 *
 * https://docs.aws.amazon.com/codepipeline/latest/userguide/limits.html
 */
const VALID_IDENTIFIER_REGEX = /^[a-zA-Z0-9.@_-]{1,100}$/;
/**
 * Validate the given name of a pipeline component. Pipeline component names all have the same restrictions.
 * This can be used to validate the name of all components of a pipeline.
 */
function validateName(thing, name) {
    validateAgainstRegex(VALID_IDENTIFIER_REGEX, thing, name);
}
exports.validateName = validateName;
function validateArtifactName(artifactName) {
    // https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_Artifact.html#CodePipeline-Type-Artifact-name
    validateAgainstRegex(/^[a-zA-Z0-9_-]{1,100}$/, 'Artifact', artifactName);
}
exports.validateArtifactName = validateArtifactName;
function validateNamespaceName(namespaceName) {
    validateAgainstRegex(/^[A-Za-z0-9@_-]{1,100}$/, 'Namespace', namespaceName);
}
exports.validateNamespaceName = validateNamespaceName;
function validateAgainstRegex(regex, thing, name) {
    // name could be a Token - in that case, skip validation altogether
    if (cdk.Token.isUnresolved(name)) {
        return;
    }
    if (name !== undefined && !regex.test(name)) {
        throw new Error(`${thing} name must match regular expression: ${regex.toString()}, got '${name}'`);
    }
}
//# sourceMappingURL=data:application/json;base64,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