"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
            targetKey: encryptionKey,
            aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryptionKey: encryptionAlias,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default');
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,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