"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const case_1 = require("case");
const action_1 = require("../action");
const common_1 = require("../common");
// Class copied verbatim from the aws-s3-deployment module.
// Yes, it sucks that we didn't abstract this properly in a common class,
// but having 2 different CacheControl classes that behave differently would be worse I think.
// Something to do when CDK 2.0.0 comes out.
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * Use the provided static factory methods to construct instances of this class.
 * Used in the {@link S3DeployActionProps.cacheControl} property.
 *
 * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9
 */
class CacheControl {
    /** @param value the actual text value of the created directive */
    constructor(value) {
        this.value = value;
    }
    /** The 'must-revalidate' cache control directive. */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /** The 'no-cache' cache control directive. */
    static noCache() { return new CacheControl('no-cache'); }
    /** The 'no-transform' cache control directive. */
    static noTransform() { return new CacheControl('no-transform'); }
    /** The 'public' cache control directive. */
    static setPublic() { return new CacheControl('public'); }
    /** The 'private' cache control directive. */
    static setPrivate() { return new CacheControl('private'); }
    /** The 'proxy-revalidate' cache control directive. */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /** The 'max-age' cache control directive. */
    static maxAge(t) { return new CacheControl(`max-age: ${t.toSeconds()}`); }
    /** The 's-max-age' cache control directive. */
    static sMaxAge(t) { return new CacheControl(`s-maxage: ${t.toSeconds()}`); }
    /**
     * Allows you to create an arbitrary cache control directive,
     * in case our support is missing a method for a particular directive.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Deploys the sourceArtifact to Amazon S3.
 */
class S3DeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'S3',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [props.input],
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // pipeline needs permissions to write to the S3 bucket
        this.props.bucket.grantWrite(options.role);
        // the Action Role also needs to read from the Pipeline's bucket
        options.bucket.grantRead(options.role);
        const acl = this.props.accessControl;
        return {
            configuration: {
                BucketName: this.props.bucket.bucketName,
                Extract: this.props.extract === false ? 'false' : 'true',
                ObjectKey: this.props.objectKey,
                CannedACL: acl ? case_1.kebab(acl.toString()) : undefined,
                CacheControl: this.props.cacheControl && this.props.cacheControl.map(ac => ac.value).join(', '),
            },
        };
    }
}
exports.S3DeployAction = S3DeployAction;
//# sourceMappingURL=data:application/json;base64,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