"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const targets = require("../../../aws-events-targets"); // Automatically re-written from '@aws-cdk/aws-events-targets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * The ECR Repository source CodePipeline Action.
 *
 * Will trigger the pipeline as soon as the target tag in the repository
 * changes, but only if there is a CloudTrail Trail in the account that
 * captures the ECR event.
 */
class EcrSourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'ECR',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            registryId: this.variableExpression('RegistryId'),
            repositoryName: this.variableExpression('RepositoryName'),
            imageDigest: this.variableExpression('ImageDigest'),
            imageTag: this.variableExpression('ImageTag'),
            imageUri: this.variableExpression('ImageURI'),
        };
    }
    bound(_scope, stage, options) {
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['ecr:DescribeImages'],
            resources: [this.props.repository.repositoryArn],
        }));
        this.props.repository.onCloudTrailImagePushed(stage.pipeline.node.uniqueId + 'SourceEventRule', {
            target: new targets.CodePipeline(stage.pipeline),
            imageTag: this.props.imageTag,
        });
        // the Action Role also needs to write to the Pipeline's bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                ImageTag: this.props.imageTag,
            },
        };
    }
}
exports.EcrSourceAction = EcrSourceAction;
//# sourceMappingURL=data:application/json;base64,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