"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const action_1 = require("../action");
class CodeDeployEcsDeployAction extends action_1.Action {
    constructor(props) {
        const inputs = [];
        inputs.push(determineTaskDefinitionArtifact(props));
        inputs.push(determineAppSpecArtifact(props));
        if (props.containerImageInputs) {
            if (props.containerImageInputs.length > 4) {
                throw new Error(`Action cannot have more than 4 container image inputs, got: ${props.containerImageInputs.length}`);
            }
            for (const imageInput of props.containerImageInputs) {
                inputs.push(imageInput.input);
            }
        }
        super({
            ...props,
            resource: props.deploymentGroup,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'CodeDeployToECS',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 0 },
            inputs,
        });
        this.actionProps = props;
    }
    bound(_scope, _stage, options) {
        // permissions, based on:
        // https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.application.applicationArn],
            actions: ['codedeploy:GetApplication', 'codedeploy:GetApplicationRevision', 'codedeploy:RegisterApplicationRevision'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentGroupArn],
            actions: ['codedeploy:CreateDeployment', 'codedeploy:GetDeployment'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentConfig.deploymentConfigArn],
            actions: ['codedeploy:GetDeploymentConfig'],
        }));
        // Allow action to register the task definition template file with ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['ecs:RegisterTaskDefinition'],
        }));
        // Allow passing any roles specified in the task definition template file to ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ecs-tasks.amazonaws.com',
                    ],
                },
            },
        }));
        // the Action's Role needs to read from the Bucket to get artifacts
        options.bucket.grantRead(options.role);
        const actionConfig = {
            configuration: {
                ApplicationName: this.actionProps.deploymentGroup.application.applicationName,
                DeploymentGroupName: this.actionProps.deploymentGroup.deploymentGroupName,
                TaskDefinitionTemplateArtifact: determineTaskDefinitionArtifact(this.actionProps).artifactName,
                TaskDefinitionTemplatePath: this.actionProps.taskDefinitionTemplateFile
                    ? this.actionProps.taskDefinitionTemplateFile.fileName
                    : 'taskdef.json',
                AppSpecTemplateArtifact: determineAppSpecArtifact(this.actionProps).artifactName,
                AppSpecTemplatePath: this.actionProps.appSpecTemplateFile
                    ? this.actionProps.appSpecTemplateFile.fileName
                    : 'appspec.yaml',
            },
        };
        if (this.actionProps.containerImageInputs) {
            for (let i = 1; i <= this.actionProps.containerImageInputs.length; i++) {
                const imageInput = this.actionProps.containerImageInputs[i - 1];
                actionConfig.configuration[`Image${i}ArtifactName`] = imageInput.input.artifactName;
                actionConfig.configuration[`Image${i}ContainerName`] = imageInput.taskDefinitionPlaceholder
                    ? imageInput.taskDefinitionPlaceholder
                    : 'IMAGE';
            }
        }
        return actionConfig;
    }
}
exports.CodeDeployEcsDeployAction = CodeDeployEcsDeployAction;
function determineTaskDefinitionArtifact(props) {
    if (props.taskDefinitionTemplateFile && props.taskDefinitionTemplateInput) {
        throw new Error("Exactly one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.taskDefinitionTemplateFile) {
        return props.taskDefinitionTemplateFile.artifact;
    }
    if (props.taskDefinitionTemplateInput) {
        return props.taskDefinitionTemplateInput;
    }
    throw new Error("Specifying one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' is required for the ECS CodeDeploy Action");
}
function determineAppSpecArtifact(props) {
    if (props.appSpecTemplateFile && props.appSpecTemplateInput) {
        throw new Error("Exactly one of 'appSpecTemplateInput' or 'appSpecTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.appSpecTemplateFile) {
        return props.appSpecTemplateFile.artifact;
    }
    if (props.appSpecTemplateInput) {
        return props.appSpecTemplateInput;
    }
    throw new Error("Specifying one of 'appSpecTemplateInput' or 'appSpecTemplateFile' is required for the ECS CodeDeploy Action");
}
//# sourceMappingURL=data:application/json;base64,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