"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const codeguruprofiler_generated_1 = require("./codeguruprofiler.generated");
class ProfilingGroupBase extends core_1.Resource {
    /**
     * Grant access to publish profiling information to the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:ConfigureAgent
     *  - codeguru-profiler:PostAgentProfile
     *
     * @param grantee Principal to grant publish rights to
     */
    grantPublish(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:ConfigureAgent', 'codeguru-profiler:PostAgentProfile'],
            resourceArns: [this.profilingGroupArn],
        });
    }
    /**
     * Grant access to read profiling information from the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:GetProfile
     *  - codeguru-profiler:DescribeProfilingGroup
     *
     * @param grantee Principal to grant read rights to
     */
    grantRead(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:GetProfile', 'codeguru-profiler:DescribeProfilingGroup'],
            resourceArns: [this.profilingGroupArn],
        });
    }
}
/**
 * A new Profiling Group.
 */
class ProfilingGroup extends ProfilingGroupBase {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: (_a = props.profilingGroupName) !== null && _a !== void 0 ? _a : core_1.Lazy.stringValue({ produce: () => this.generateUniqueId() }),
        });
        const profilingGroup = new codeguruprofiler_generated_1.CfnProfilingGroup(this, 'ProfilingGroup', {
            profilingGroupName: this.physicalName,
        });
        this.profilingGroupName = this.getResourceNameAttribute(profilingGroup.ref);
        this.profilingGroupArn = this.getResourceArnAttribute(profilingGroup.attrArn, {
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing Profiling Group provided a Profiling Group Name.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupName Profiling Group Name
     */
    static fromProfilingGroupName(scope, id, profilingGroupName) {
        const stack = core_1.Stack.of(scope);
        return this.fromProfilingGroupArn(scope, id, stack.formatArn({
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: profilingGroupName,
        }));
    }
    /**
     * Import an existing Profiling Group provided an ARN.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupArn Profiling Group ARN
     */
    static fromProfilingGroupArn(scope, id, profilingGroupArn) {
        class Import extends ProfilingGroupBase {
            constructor() {
                super(...arguments);
                this.profilingGroupName = core_1.Stack.of(scope).parseArn(profilingGroupArn).resource;
                this.profilingGroupArn = profilingGroupArn;
            }
        }
        return new Import(scope, id);
    }
    generateUniqueId() {
        const name = this.node.uniqueId;
        if (name.length > 240) {
            return name.substring(0, 120) + name.substring(name.length - 120);
        }
        return name;
    }
}
exports.ProfilingGroup = ProfilingGroup;
//# sourceMappingURL=data:application/json;base64,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