"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const METRICKEY_SYMBOL = Symbol('@aws-cdk/aws-cloudwatch.MetricKey');
/**
 * Return a unique string representation for this metric.
 *
 * Can be used to determine as a hash key to determine if 2 Metric objects
 * represent the same metric. Excludes rendering properties.
 */
function metricKey(metric) {
    // Cache on the object itself. This is safe because Metric objects are immutable.
    if (metric.hasOwnProperty(METRICKEY_SYMBOL)) {
        return metric[METRICKEY_SYMBOL];
    }
    const parts = new Array();
    const conf = metric.toMetricConfig();
    if (conf.mathExpression) {
        parts.push(conf.mathExpression.expression);
        for (const id of Object.keys(conf.mathExpression.usingMetrics).sort()) {
            parts.push(id);
            parts.push(metricKey(conf.mathExpression.usingMetrics[id]));
        }
    }
    if (conf.metricStat) {
        parts.push(conf.metricStat.namespace);
        parts.push(conf.metricStat.metricName);
        for (const dim of conf.metricStat.dimensions || []) {
            parts.push(dim.name);
            parts.push(dim.value);
        }
        if (conf.metricStat.statistic) {
            parts.push(conf.metricStat.statistic);
        }
        if (conf.metricStat.period) {
            parts.push(`${conf.metricStat.period.toSeconds()}`);
        }
        if (conf.metricStat.region) {
            parts.push(conf.metricStat.region);
        }
        if (conf.metricStat.account) {
            parts.push(conf.metricStat.account);
        }
    }
    const ret = parts.join('|');
    Object.defineProperty(metric, METRICKEY_SYMBOL, { value: ret });
    return ret;
}
exports.metricKey = metricKey;
/**
 * Return the period of a metric
 *
 * For a stat metric, return the immediate period.
 *
 * For an expression metric, all metrics used in it have been made to have the
 * same period, so we return the period of the first inner metric.
 */
function metricPeriod(metric) {
    return dispatchMetric(metric, {
        withStat(stat) {
            return stat.period;
        },
        withExpression() {
            return metric.period || core_1.Duration.minutes(5);
        },
    });
}
exports.metricPeriod = metricPeriod;
/**
 * Given a metric object, inspect it and call the correct function for the type of output
 *
 * In addition to the metric object itself, takes a callback object with two
 * methods, to be invoked for the particular type of metric.
 *
 * If the metric represent a metric query (nominally generated through an
 * instantiation of `Metric` but can be generated by any class that implements
 * `IMetric`) a particular field in its `toMetricConfig()` output will be set
 * (to wit, `metricStat`) and the `withStat()` callback will be called with
 * that object.
 *
 * If the metric represents an expression (usually by instantiating `MathExpression`
 * but users can implement `IMetric` arbitrarily) the `mathExpression` field
 * will be set in the object returned from `toMetricConfig` and the callback
 * called `withExpression` will be applied to that object.
 *
 * Will return the values returned by the callbacks.
 *
 * To be used as such:
 *
 * ```ts
 * const ret = dispatchMetric(someMetric, {
 *   withStat(stat) {
 *     // do something with stat
 *     return 1;
 *   },
 *   withExpression(expr) {
 *     // do something with expr
 *     return 2;
 *   },
 * });
 * ```
 *
 * This function encapsulates some type analysis that would otherwise have to be
 * repeated in all places where code needs to make a distinction on the type
 * of metric object that is being passed.
 */
// tslint:disable-next-line:max-line-length
function dispatchMetric(metric, fns) {
    const conf = metric.toMetricConfig();
    if (conf.metricStat && conf.mathExpression) {
        throw new Error('Metric object must not produce both \'metricStat\' and \'mathExpression\'');
    }
    else if (conf.metricStat) {
        return fns.withStat(conf.metricStat, conf);
    }
    else if (conf.mathExpression) {
        return fns.withExpression(conf.mathExpression, conf);
    }
    else {
        throw new Error('Metric object must have either \'metricStat\' or \'mathExpression\'');
    }
}
exports.dispatchMetric = dispatchMetric;
//# sourceMappingURL=data:application/json;base64,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