"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const alarm_1 = require("./alarm");
const metric_util_1 = require("./private/metric-util");
const statistic_1 = require("./private/statistic");
/**
 * A metric emitted by a service
 *
 * The metric is a combination of a metric identifier (namespace, name and dimensions)
 * and an aggregation function (statistic, period and unit).
 *
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * Metric is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class Metric {
    constructor(props) {
        this.period = props.period || cdk.Duration.minutes(5);
        const periodSec = this.period.toSeconds();
        if (periodSec !== 1 && periodSec !== 5 && periodSec !== 10 && periodSec !== 30 && periodSec % 60 !== 0) {
            throw new Error(`'period' must be 1, 5, 10, 30, or a multiple of 60 seconds, received ${periodSec}`);
        }
        this.dimensions = props.dimensions;
        this.namespace = props.namespace;
        this.metricName = props.metricName;
        // Try parsing, this will throw if it's not a valid stat
        this.statistic = statistic_1.normalizeStatistic(props.statistic || 'Average');
        this.label = props.label;
        this.color = props.color;
        this.unit = props.unit;
        this.account = props.account;
        this.region = props.region;
    }
    /**
     * Grant permissions to the given identity to write metrics.
     *
     * @param grantee The IAM identity to give permissions to.
     */
    static grantPutMetricData(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['cloudwatch:PutMetricData'],
            resourceArns: ['*'],
        });
    }
    /**
     * Return a copy of Metric `with` properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.statistic === undefined || props.statistic === this.statistic)
            && (props.unit === undefined || props.unit === this.unit)
            && (props.account === undefined || props.account === this.account)
            && (props.region === undefined || props.region === this.region)
            // For these we're not going to do deep equality, misses some opportunity for optimization
            // but that's okay.
            && (props.dimensions === undefined)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())) {
            return this;
        }
        return new Metric({
            dimensions: ifUndefined(props.dimensions, this.dimensions),
            namespace: this.namespace,
            metricName: this.metricName,
            period: ifUndefined(props.period, this.period),
            statistic: ifUndefined(props.statistic, this.statistic),
            unit: ifUndefined(props.unit, this.unit),
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            account: ifUndefined(props.account, this.account),
            region: ifUndefined(props.region, this.region),
        });
    }
    /**
     * Attach the metric object to the given construct scope
     *
     * Returns a Metric object that uses the account and region from the Stack
     * the given construct is defined in. If the metric is subsequently used
     * in a Dashboard or Alarm in a different Stack defined in a different
     * account or region, the appropriate 'region' and 'account' fields
     * will be added to it.
     *
     * If the scope we attach to is in an environment-agnostic stack,
     * nothing is done and the same Metric object is returned.
     */
    attachTo(scope) {
        const stack = cdk.Stack.of(scope);
        return this.with({
            region: cdk.Token.isUnresolved(stack.region) ? undefined : stack.region,
            account: cdk.Token.isUnresolved(stack.account) ? undefined : stack.account,
        });
    }
    toMetricConfig() {
        const dims = this.dimensionsAsList();
        return {
            metricStat: {
                dimensions: dims.length > 0 ? dims : undefined,
                namespace: this.namespace,
                metricName: this.metricName,
                period: this.period,
                statistic: this.statistic,
                unitFilter: this.unit,
                account: this.account,
                region: this.region,
            },
            renderingProperties: {
                color: this.color,
                label: this.label,
            },
        };
    }
    toAlarmConfig() {
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        const stat = statistic_1.parseStatistic(metricConfig.metricStat.statistic);
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: stat.type === 'simple' ? stat.statistic : undefined,
            extendedStatistic: stat.type === 'percentile' ? 'p' + stat.percentile : undefined,
            unit: this.unit,
        };
    }
    toGraphConfig() {
        var _a, _b, _c, _d;
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            renderingProperties: {
                period: metricConfig.metricStat.period.toSeconds(),
                stat: metricConfig.metricStat.statistic,
                color: asString((_a = metricConfig.renderingProperties) === null || _a === void 0 ? void 0 : _a.color),
                label: asString((_b = metricConfig.renderingProperties) === null || _b === void 0 ? void 0 : _b.label),
            },
            // deprecated properties for backwards compatibility
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: metricConfig.metricStat.statistic,
            color: asString((_c = metricConfig.renderingProperties) === null || _c === void 0 ? void 0 : _c.color),
            label: asString((_d = metricConfig.renderingProperties) === null || _d === void 0 ? void 0 : _d.label),
            unit: this.unit,
        };
    }
    /**
     * Make a new Alarm for this metric
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                statistic: props.statistic,
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    toString() {
        return this.label || this.metricName;
    }
    /**
     * Return the dimensions of this Metric as a list of Dimension.
     */
    dimensionsAsList() {
        const dims = this.dimensions;
        if (dims === undefined) {
            return [];
        }
        const list = Object.keys(dims).sort().map(key => ({ name: key, value: dims[key] }));
        return list;
    }
}
exports.Metric = Metric;
function asString(x) {
    if (x === undefined) {
        return undefined;
    }
    if (typeof x !== 'string') {
        throw new Error(`Expected string, got ${x}`);
    }
    return x;
}
/**
 * A math expression built with metric(s) emitted by a service
 *
 * The math expression is a combination of an expression (x+y) and metrics to apply expression on.
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * MathExpression is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class MathExpression {
    constructor(props) {
        this.period = props.period || cdk.Duration.minutes(5);
        this.expression = props.expression;
        this.usingMetrics = changeAllPeriods(props.usingMetrics, this.period);
        this.label = props.label;
        this.color = props.color;
        const invalidVariableNames = Object.keys(props.usingMetrics).filter(x => !validVariableName(x));
        if (invalidVariableNames.length > 0) {
            throw new Error(`Invalid variable names in expression: ${invalidVariableNames}. Must start with lowercase letter and only contain alphanumerics.`);
        }
        this.validateNoIdConflicts();
    }
    /**
     * Return a copy of Metric with properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())) {
            return this;
        }
        return new MathExpression({
            expression: this.expression,
            usingMetrics: this.usingMetrics,
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            period: ifUndefined(props.period, this.period),
        });
    }
    toAlarmConfig() {
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    toGraphConfig() {
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    toMetricConfig() {
        return {
            mathExpression: {
                period: this.period.toSeconds(),
                expression: this.expression,
                usingMetrics: this.usingMetrics,
            },
            renderingProperties: {
                label: this.label,
                color: this.color,
            },
        };
    }
    /**
     * Make a new Alarm for this metric
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    toString() {
        return this.label || this.expression;
    }
    validateNoIdConflicts() {
        const seen = new Map();
        visit(this);
        function visit(metric) {
            metric_util_1.dispatchMetric(metric, {
                withStat() {
                    // Nothing
                },
                withExpression(expr) {
                    for (const [id, subMetric] of Object.entries(expr.usingMetrics)) {
                        const existing = seen.get(id);
                        if (existing && metric_util_1.metricKey(existing) !== metric_util_1.metricKey(subMetric)) {
                            throw new Error(`The ID '${id}' used for two metrics in the expression: '${subMetric}' and '${existing}'. Rename one.`);
                        }
                        seen.set(id, subMetric);
                        visit(subMetric);
                    }
                },
            });
        }
    }
}
exports.MathExpression = MathExpression;
const VALID_VARIABLE = new RegExp('^[a-z][a-zA-Z0-9_]*$');
function validVariableName(x) {
    return VALID_VARIABLE.test(x);
}
function ifUndefined(x, def) {
    if (x !== undefined) {
        return x;
    }
    return def;
}
/**
 * Change periods of all metrics in the map
 */
function changeAllPeriods(metrics, period) {
    const ret = {};
    for (const [id, metric] of Object.entries(metrics)) {
        ret[id] = changePeriod(metric, period);
    }
    return ret;
}
/**
 * Return a new metric object which is the same type as the input object, but with the period changed
 *
 * Relies on the fact that implementations of `IMetric` are also supposed to have
 * an implementation of `with` that accepts an argument called `period`. See `IModifiableMetric`.
 */
function changePeriod(metric, period) {
    if (isModifiableMetric(metric)) {
        return metric.with({ period });
    }
    throw new Error(`Metric object should also implement 'with': ${metric}`);
}
function isModifiableMetric(m) {
    return typeof m === 'object' && m !== null && !!m.with;
}
//# sourceMappingURL=data:application/json;base64,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