"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rendering_1 = require("./private/rendering");
const widget_1 = require("./widget");
/**
 * Display the metric associated with an alarm, including the alarm line
 */
class AlarmWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    annotations: {
                        alarms: [this.props.alarm.alarmArn],
                    },
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                    },
                },
            }];
    }
}
exports.AlarmWidget = AlarmWidget;
/**
 * A dashboard widget that displays metrics
 */
class GraphWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        const horizontalAnnotations = [
            ...(this.props.leftAnnotations || []).map(mapAnnotation('left')),
            ...(this.props.rightAnnotations || []).map(mapAnnotation('right')),
        ];
        const metrics = rendering_1.allMetricsGraphJson(this.props.left || [], this.props.right || []);
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    stacked: this.props.stacked,
                    metrics: metrics.length > 0 ? metrics : undefined,
                    annotations: horizontalAnnotations.length > 0 ? { horizontal: horizontalAnnotations } : undefined,
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                        right: this.props.rightYAxis !== undefined ? this.props.rightYAxis : undefined,
                    },
                    legend: this.props.legendPosition !== undefined ? { position: this.props.legendPosition } : undefined,
                },
            }];
    }
}
exports.GraphWidget = GraphWidget;
/**
 * A dashboard widget that displays the most recent value for every metric
 */
class SingleValueWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 3);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'singleValue',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    metrics: rendering_1.allMetricsGraphJson(this.props.metrics, []),
                    setPeriodToTimeRange: this.props.setPeriodToTimeRange,
                },
            }];
    }
}
exports.SingleValueWidget = SingleValueWidget;
/**
 * Fill shading options that will be used with an annotation
 */
var Shading;
(function (Shading) {
    /**
     * Don't add shading
     */
    Shading["NONE"] = "none";
    /**
     * Add shading above the annotation
     */
    Shading["ABOVE"] = "above";
    /**
     * Add shading below the annotation
     */
    Shading["BELOW"] = "below";
})(Shading = exports.Shading || (exports.Shading = {}));
/**
 * A set of standard colours that can be used in annotations in a GraphWidget.
 */
class Color {
}
exports.Color = Color;
/** blue - hex #1f77b4 */
Color.BLUE = '#1f77b4';
/** brown - hex #8c564b */
Color.BROWN = '#8c564b';
/** green - hex #2ca02c */
Color.GREEN = '#2ca02c';
/** grey - hex #7f7f7f */
Color.GREY = '#7f7f7f';
/** orange - hex #ff7f0e */
Color.ORANGE = '#ff7f0e';
/** pink - hex #e377c2 */
Color.PINK = '#e377c2';
/** purple - hex #9467bd */
Color.PURPLE = '#9467bd';
/** red - hex #d62728 */
Color.RED = '#d62728';
/**
 * The position of the legend on a GraphWidget.
 */
var LegendPosition;
(function (LegendPosition) {
    /**
     * Legend appears below the graph (default).
     */
    LegendPosition["BOTTOM"] = "bottom";
    /**
     * Add shading above the annotation
     */
    LegendPosition["RIGHT"] = "right";
    /**
     * Add shading below the annotation
     */
    LegendPosition["HIDDEN"] = "hidden";
})(LegendPosition = exports.LegendPosition || (exports.LegendPosition = {}));
function mapAnnotation(yAxis) {
    return (a) => {
        return { ...a, yAxis };
    };
}
//# sourceMappingURL=data:application/json;base64,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