"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloud9_generated_1 = require("../lib/cloud9.generated");
/**
 * A Cloud9 Environment with Amazon EC2
 * @resource AWS::Cloud9::EnvironmentEC2
 */
class Ec2Environment extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.vpc = props.vpc;
        if (!props.subnetSelection && this.vpc.publicSubnets.length === 0) {
            throw new Error('no subnetSelection specified and no public subnet found in the vpc, please specify subnetSelection');
        }
        const vpcSubnets = (_a = props.subnetSelection) !== null && _a !== void 0 ? _a : { subnetType: ec2.SubnetType.PUBLIC };
        const c9env = new cloud9_generated_1.CfnEnvironmentEC2(this, 'Resource', {
            name: props.ec2EnvironmentName,
            description: props.description,
            instanceType: (_c = (_b = props.instanceType) === null || _b === void 0 ? void 0 : _b.toString()) !== null && _c !== void 0 ? _c : ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.MICRO).toString(),
            subnetId: this.vpc.selectSubnets(vpcSubnets).subnetIds[0],
            repositories: props.clonedRepositories ? props.clonedRepositories.map(r => ({
                repositoryUrl: r.repositoryUrl,
                pathComponent: r.pathComponent,
            })) : undefined,
        });
        this.environmentId = c9env.ref;
        this.ec2EnvironmentArn = c9env.getAtt('Arn').toString();
        this.ec2EnvironmentName = c9env.getAtt('Name').toString();
        this.ideUrl = `https://${this.stack.region}.console.aws.amazon.com/cloud9/ide/${this.environmentId}`;
    }
    /**
     * import from EnvironmentEc2Name
     */
    static fromEc2EnvironmentName(scope, id, ec2EnvironmentName) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.ec2EnvironmentName = ec2EnvironmentName;
                this.ec2EnvironmentArn = cdk.Stack.of(this).formatArn({
                    service: 'cloud9',
                    resource: 'environment',
                    resourceName: this.ec2EnvironmentName,
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.Ec2Environment = Ec2Environment;
/**
 * The class for different repository providers
 */
class CloneRepository {
    constructor(repositoryUrl, pathComponent) {
        this.repositoryUrl = repositoryUrl;
        this.pathComponent = pathComponent;
    }
    /**
     * import repository to cloud9 environment from AWS CodeCommit
     *
     * @param repository the codecommit repository to clone from
     * @param path  the target path in cloud9 environment
     */
    static fromCodeCommit(repository, path) {
        return {
            repositoryUrl: repository.repositoryCloneUrlHttp,
            pathComponent: path,
        };
    }
}
exports.CloneRepository = CloneRepository;
//# sourceMappingURL=data:application/json;base64,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