"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const backup_generated_1 = require("./backup.generated");
/**
 * Backup vault events
 */
var BackupVaultEvents;
(function (BackupVaultEvents) {
    /** BACKUP_JOB_STARTED */
    BackupVaultEvents["BACKUP_JOB_STARTED"] = "BACKUP_JOB_STARTED";
    /** BACKUP_JOB_COMPLETED */
    BackupVaultEvents["BACKUP_JOB_COMPLETED"] = "BACKUP_JOB_COMPLETED";
    /** BACKUP_JOB_SUCCESSFUL */
    BackupVaultEvents["BACKUP_JOB_SUCCESSFUL"] = "BACKUP_JOB_SUCCESSFUL";
    /** BACKUP_JOB_FAILED */
    BackupVaultEvents["BACKUP_JOB_FAILED"] = "BACKUP_JOB_FAILED";
    /** BACKUP_JOB_EXPIRED */
    BackupVaultEvents["BACKUP_JOB_EXPIRED"] = "BACKUP_JOB_EXPIRED";
    /** RESTORE_JOB_STARTED */
    BackupVaultEvents["RESTORE_JOB_STARTED"] = "RESTORE_JOB_STARTED";
    /** RESTORE_JOB_COMPLETED */
    BackupVaultEvents["RESTORE_JOB_COMPLETED"] = "RESTORE_JOB_COMPLETED";
    /** RESTORE_JOB_SUCCESSFUL */
    BackupVaultEvents["RESTORE_JOB_SUCCESSFUL"] = "RESTORE_JOB_SUCCESSFUL";
    /** RESTORE_JOB_FAILED */
    BackupVaultEvents["RESTORE_JOB_FAILED"] = "RESTORE_JOB_FAILED";
    /** COPY_JOB_STARTED */
    BackupVaultEvents["COPY_JOB_STARTED"] = "COPY_JOB_STARTED";
    /** COPY_JOB_SUCCESSFUL */
    BackupVaultEvents["COPY_JOB_SUCCESSFUL"] = "COPY_JOB_SUCCESSFUL";
    /** COPY_JOB_FAILED */
    BackupVaultEvents["COPY_JOB_FAILED"] = "COPY_JOB_FAILED";
    /** RECOVERY_POINT_MODIFIED */
    BackupVaultEvents["RECOVERY_POINT_MODIFIED"] = "RECOVERY_POINT_MODIFIED";
    /** BACKUP_PLAN_CREATED */
    BackupVaultEvents["BACKUP_PLAN_CREATED"] = "BACKUP_PLAN_CREATED";
    /** BACKUP_PLAN_MODIFIED */
    BackupVaultEvents["BACKUP_PLAN_MODIFIED"] = "BACKUP_PLAN_MODIFIED";
})(BackupVaultEvents = exports.BackupVaultEvents || (exports.BackupVaultEvents = {}));
/**
 * A backup vault
 */
class BackupVault extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.backupVaultName && !/^[a-zA-Z0-9\-_\.]{1,50}$/.test(props.backupVaultName)) {
            throw new Error('Expected vault name to match pattern `^[a-zA-Z0-9\-\_\.]{1,50}$`');
        }
        let notifications;
        if (props.notificationTopic) {
            notifications = {
                backupVaultEvents: props.notificationEvents || Object.values(BackupVaultEvents),
                snsTopicArn: props.notificationTopic.topicArn,
            };
            props.notificationTopic.grantPublish(new iam.ServicePrincipal('backup.amazonaws.com'));
        }
        const vault = new backup_generated_1.CfnBackupVault(this, 'Resource', {
            backupVaultName: props.backupVaultName || this.uniqueVaultName(),
            accessPolicy: props.accessPolicy && props.accessPolicy.toJSON(),
            encryptionKeyArn: props.encryptionKey && props.encryptionKey.keyArn,
            notifications,
        });
        vault.applyRemovalPolicy(props.removalPolicy);
        this.backupVaultName = vault.attrBackupVaultName;
        this.backupVaultArn = vault.attrBackupVaultArn;
    }
    /**
     * Import an existing backup vault
     */
    static fromBackupVaultName(scope, id, backupVaultName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.backupVaultName = backupVaultName;
            }
        }
        return new Import(scope, id);
    }
    uniqueVaultName() {
        // Max length of 50 chars, get the last 50 chars
        const id = `${this.node.uniqueId}${core_1.Aws.STACK_NAME}`;
        return id.substring(Math.max(id.length - 50, 0), id.length);
    }
}
exports.BackupVault = BackupVault;
//# sourceMappingURL=data:application/json;base64,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