"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_autoscaling_common_1 = require("../../aws-autoscaling-common"); // Automatically re-written from '@aws-cdk/aws-autoscaling-common'
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const step_scaling_action_1 = require("./step-scaling-action");
/**
 * Define a acaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 */
class StepScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = aws_autoscaling_common_1.normalizeIntervals(props.scalingSteps, changesAreAbsolute);
        const alarms = aws_autoscaling_common_1.findAlarmThresholds(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                autoScalingGroup: props.autoScalingGroup,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                autoScalingGroup: props.autoScalingGroup,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined,
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
exports.StepScalingPolicy = StepScalingPolicy;
function aggregationTypeFromMetric(metric) {
    var _a;
    const statistic = (_a = metric.toMetricConfig().metricStat) === null || _a === void 0 ? void 0 : _a.statistic;
    if (statistic === undefined) {
        return undefined;
    } // Math expression, don't know aggregation, leave default
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            throw new Error(`Cannot only scale on 'Minimum', 'Maximum', 'Average' metrics, got ${statistic}`);
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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