"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        const resource = new autoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyType: 'StepScaling',
            autoScalingGroupName: props.autoScalingGroup.autoScalingGroupName,
            cooldown: props.cooldown && props.cooldown.toSeconds().toString(),
            estimatedInstanceWarmup: props.estimatedInstanceWarmup && props.estimatedInstanceWarmup.toSeconds(),
            adjustmentType: props.adjustmentType,
            minAdjustmentMagnitude: props.minAdjustmentMagnitude,
            metricAggregationType: props.metricAggregationType,
            stepAdjustments: cdk.Lazy.anyValue({ produce: () => this.adjustments }),
        });
        this.scalingPolicyArn = resource.ref;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["CHANGE_IN_CAPACITY"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["EXACT_CAPACITY"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["AVERAGE"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["MINIMUM"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["MAXIMUM"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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