"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const route_1 = require("./route");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualRouterBase extends cdk.Resource {
    /**
     * Add a single route to the router
     */
    addRoute(id, props) {
        const route = new route_1.Route(this, id, {
            ...props,
            routeName: id,
            mesh: this.mesh,
            virtualRouter: this,
        });
        return route;
    }
}
class VirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualRouterName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.listeners = new Array();
        this.mesh = props.mesh;
        this.addListener(props.listener || { portMapping: { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP } });
        const router = new appmesh_generated_1.CfnVirtualRouter(this, 'Resource', {
            virtualRouterName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners,
            },
        });
        this.virtualRouterName = this.getResourceNameAttribute(router.attrVirtualRouterName);
        this.virtualRouterArn = this.getResourceArnAttribute(router.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualRouter given an ARN
     */
    static fromVirtualRouterArn(scope, id, virtualRouterArn) {
        return new ImportedVirtualRouter(scope, id, { virtualRouterArn });
    }
    /**
     * Import an existing VirtualRouter given names
     */
    static fromVirtualRouterName(scope, id, meshName, virtualRouterName) {
        return new ImportedVirtualRouter(scope, id, { meshName, virtualRouterName });
    }
    /**
     * Import an existing virtual router given attributes
     */
    static fromVirtualRouterAttributes(scope, id, attrs) {
        return new ImportedVirtualRouter(scope, id, attrs);
    }
    /**
     * Add port mappings to the router
     */
    addListener(listener) {
        this.listeners.push({
            portMapping: listener.portMapping,
        });
    }
}
exports.VirtualRouter = VirtualRouter;
/**
 * Used to import a VirtualRouter and perform actions or read properties from
 */
class ImportedVirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.mesh) {
            this._mesh = props.mesh;
        }
        if (props.meshName) {
            if (props.mesh) {
                throw new Error('Supply either \'mesh\' or \'meshName\', but not both');
            }
            this._mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', props.meshName);
        }
        if (props.virtualRouterArn) {
            this.virtualRouterArn = props.virtualRouterArn;
            this.virtualRouterName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualRouterArn).resourceName));
        }
        else if (props.virtualRouterName && props.meshName) {
            this.virtualRouterName = props.virtualRouterName;
            this.virtualRouterArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter`,
                resourceName: this.virtualRouterName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
    /**
     * The AppMesh mesh the VirtualRouter belongs to
     */
    get mesh() {
        if (!this._mesh) {
            throw new Error('Please supply either \'mesh\' or \'meshName\' when calling \'fromVirtualRouterAttributes\'');
        }
        return this._mesh;
    }
}
//# sourceMappingURL=data:application/json;base64,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