"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * Type of route
 */
var RouteType;
(function (RouteType) {
    /**
     * HTTP route
     */
    RouteType["HTTP"] = "http";
    /**
     * TCP route
     */
    RouteType["TCP"] = "tcp";
})(RouteType = exports.RouteType || (exports.RouteType = {}));
/**
 * Route represents a new or existing route attached to a VirtualRouter and Mesh
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
 */
class Route extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.routeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.weightedTargets = new Array();
        this.virtualRouter = props.virtualRouter;
        const routeType = props.routeType !== undefined ? props.routeType :
            props.prefix !== undefined ? RouteType.HTTP :
                RouteType.TCP;
        if (routeType === RouteType.HTTP) {
            this.httpRoute = this.renderHttpRoute(props);
        }
        else {
            this.tcpRoute = this.renderTcpRoute(props);
        }
        const route = new appmesh_generated_1.CfnRoute(this, 'Resource', {
            routeName: this.physicalName,
            meshName: this.virtualRouter.mesh.meshName,
            virtualRouterName: this.virtualRouter.virtualRouterName,
            spec: {
                tcpRoute: this.tcpRoute,
                httpRoute: this.httpRoute,
            },
        });
        this.routeName = this.getResourceNameAttribute(route.attrRouteName);
        this.routeArn = this.getResourceArnAttribute(route.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter/${props.virtualRouter.virtualRouterName}/route`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing route given an ARN
     */
    static fromRouteArn(scope, id, routeArn) {
        return new ImportedRoute(scope, id, { routeArn });
    }
    /**
     * Import an existing route given its name
     */
    static fromRouteName(scope, id, meshName, virtualRouterName, routeName) {
        return new ImportedRoute(scope, id, { meshName, virtualRouterName, routeName });
    }
    /**
     * Utility method to add weighted route targets to an existing route
     */
    renderWeightedTargets(props) {
        for (const t of props) {
            this.weightedTargets.push({
                virtualNode: t.virtualNode.virtualNodeName,
                weight: t.weight || 1,
            });
        }
        return this.weightedTargets;
    }
    renderHttpRoute(props) {
        return {
            match: {
                prefix: props.prefix || '/',
            },
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
    renderTcpRoute(props) {
        return {
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
}
exports.Route = Route;
/**
 * Represents and imported IRoute
 */
class ImportedRoute extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.routeArn) {
            this.routeArn = props.routeArn;
            this.routeName = cdk.Fn.select(4, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.routeArn).resourceName));
        }
        else if (props.routeName && props.meshName && props.virtualRouterName) {
            this.routeName = props.routeName;
            this.routeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter/${props.virtualRouterName}/route`,
                resourceName: this.routeName,
            });
        }
        else {
            throw new Error('Need either arn or three names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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