"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error('Exactly one of \'customMetric\' or \'predefinedMetric\' must be specified.');
        }
        if (props.customMetric && !props.customMetric.toMetricConfig().metricStat) {
            throw new Error('Only direct metrics are supported for Target Tracking. Use Step Scaling or supply a Metric object.');
        }
        super(scope, id);
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'TargetTrackingScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            targetTrackingScalingPolicyConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                scaleInCooldown: props.scaleInCooldown && props.scaleInCooldown.toSeconds(),
                scaleOutCooldown: props.scaleOutCooldown && props.scaleOutCooldown.toSeconds(),
                targetValue: props.targetValue,
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toMetricConfig().metricStat;
    if (c.statistic.startsWith('p')) {
        throw new Error(`Cannot use statistic '${c.statistic}' for Target Tracking: only 'Average', 'Minimum', 'Maximum', 'SampleCount', and 'Sum' are supported.`);
    }
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unitFilter,
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    /**
     * DYNAMODB_READ_CAPACITY_UTILIZATIO
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYNAMODB_READ_CAPACITY_UTILIZATION"] = "DynamoDBReadCapacityUtilization";
    /**
     * DYANMODB_WRITE_CAPACITY_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYANMODB_WRITE_CAPACITY_UTILIZATION"] = "DynamoDBWriteCapacityUtilization";
    /**
     * ALB_REQUEST_COUNT_PER_TARGET
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ALB_REQUEST_COUNT_PER_TARGET"] = "ALBRequestCountPerTarget";
    /**
     * RDS_READER_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_CPU_UTILIZATION"] = "RDSReaderAverageCPUUtilization";
    /**
     * RDS_READER_AVERAGE_DATABASE_CONNECTIONS
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_DATABASE_CONNECTIONS"] = "RDSReaderAverageDatabaseConnections";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION"] = "EC2SpotFleetRequestAverageCPUUtilization";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN"] = "EC2SpotFleetRequestAverageNetworkIn";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT"] = "EC2SpotFleetRequestAverageNetworkOut";
    /**
     * SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE"] = "SageMakerVariantInvocationsPerInstance";
    /**
     * ECS_SERVICE_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_CPU_UTILIZATION"] = "ECSServiceAverageCPUUtilization";
    /**
     * ECS_SERVICE_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION"] = "ECSServiceAverageMemoryUtilization";
    /**
     * LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION
     * @see https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics.html#monitoring-metrics-concurrency
     */
    PredefinedMetric["LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION"] = "LambdaProvisionedConcurrencyUtilization";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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