"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_autoscaling_common_1 = require("../../aws-autoscaling-common"); // Automatically re-written from '@aws-cdk/aws-autoscaling-common'
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const step_scaling_action_1 = require("./step-scaling-action");
/**
 * Define a acaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 */
class StepScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = aws_autoscaling_common_1.normalizeIntervals(props.scalingSteps, changesAreAbsolute);
        const alarms = aws_autoscaling_common_1.findAlarmThresholds(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined,
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
exports.StepScalingPolicy = StepScalingPolicy;
function aggregationTypeFromMetric(metric) {
    var _a;
    const statistic = (_a = metric.toMetricConfig().metricStat) === null || _a === void 0 ? void 0 : _a.statistic;
    if (statistic == null) {
        return undefined;
    } // Math expression, don't know aggregation, leave default
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            throw new Error(`Cannot only scale on 'Minimum', 'Maximum', 'Average' metrics, got ${statistic}`);
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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