"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const scalable_target_1 = require("./scalable-target");
/**
 * Represent an attribute for which autoscaling can be configured
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *   Step Scaling, so the Dynamo subclass won't expose this method).
 */
class BaseScalableAttribute extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.target = new scalable_target_1.ScalableTarget(this, 'Target', {
            serviceNamespace: this.props.serviceNamespace,
            scalableDimension: this.props.dimension,
            resourceId: this.props.resourceId,
            role: this.props.role,
            minCapacity: props.minCapacity !== undefined ? props.minCapacity : 1,
            maxCapacity: props.maxCapacity,
        });
    }
    /**
     * Scale out or in based on time
     */
    doScaleOnSchedule(id, props) {
        this.target.scaleOnSchedule(id, props);
    }
    /**
     * Scale out or in based on a metric value
     */
    doScaleOnMetric(id, props) {
        this.target.scaleOnMetric(id, props);
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    doScaleToTrackMetric(id, props) {
        this.target.scaleToTrackMetric(id, props);
    }
}
exports.BaseScalableAttribute = BaseScalableAttribute;
//# sourceMappingURL=data:application/json;base64,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