"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const integration_1 = require("./integration");
const mock_1 = require("./integrations/mock");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.resource = props.resource;
        this.restApi = props.resource.restApi;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if ((authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) && authorizationTypeOption && (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.restApi);
        }
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.restApi.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(props.integration),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: (_a = options.authorizationScopes) !== null && _a !== void 0 ? _a : defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        props.resource.restApi._attachMethod(this);
        const deployment = props.resource.restApi.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ method: methodProps });
        }
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled.
     *
     * @attribute
     */
    get methodArn() {
        if (!this.restApi.deploymentStage) {
            throw new Error(`Unable to determine ARN for method "${this.node.id}" since there is no stage associated with this API.\n` +
                'Either use the `deploy` prop or explicitly assign `deploymentStage` on the RestApi');
        }
        const stage = this.restApi.deploymentStage.stageName.toString();
        return this.restApi.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.restApi.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), 'test-invoke-stage');
    }
    renderIntegration(integration) {
        if (!integration) {
            // use defaultIntegration from API if defined
            if (this.resource.defaultIntegration) {
                return this.renderIntegration(this.resource.defaultIntegration);
            }
            // fallback to mock
            return this.renderIntegration(new mock_1.MockIntegration());
        }
        integration.bind(this);
        const options = integration._props.options || {};
        let credentials;
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === integration_1.ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === integration_1.ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // tslint:disable-next-line:max-line-length
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: integration._props.type,
            uri: integration._props.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: integration._props.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        var _a;
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.restApi.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return (_a = options.requestValidator) === null || _a === void 0 ? void 0 : _a.requestValidatorId;
    }
}
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
function pathForArn(path) {
    return path.replace(/\{[^\}]*\}/g, '*'); // replace path parameters (like '{bookId}') with asterisk
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWV0aG9kLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibWV0aG9kLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEscUNBQXdELENBQUMsZ0RBQWdEO0FBQ3pHLGlFQUFtRTtBQUNuRSw2Q0FBdUQ7QUFDdkQsK0NBQTREO0FBQzVELDhDQUFzRDtBQU10RCxpQ0FBNEM7QUFzSTVDLE1BQWEsTUFBTyxTQUFRLGVBQVE7SUFNaEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFrQjs7UUFDeEQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQztRQUN0QyxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUMsV0FBVyxFQUFFLENBQUM7UUFDakQseUJBQWtCLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ3BDLE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLElBQUksRUFBRSxDQUFDO1FBQ3BDLE1BQU0sb0JBQW9CLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsSUFBSSxFQUFFLENBQUM7UUFDdkUsTUFBTSxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsSUFBSSxvQkFBb0IsQ0FBQyxVQUFVLENBQUM7UUFDekUsTUFBTSxZQUFZLEdBQUcsVUFBVSxhQUFWLFVBQVUsdUJBQVYsVUFBVSxDQUFFLFlBQVksQ0FBQztRQUM5QyxNQUFNLHVCQUF1QixHQUFHLE9BQU8sQ0FBQyxpQkFBaUIsSUFBSSxvQkFBb0IsQ0FBQyxpQkFBaUIsQ0FBQztRQUNwRyxNQUFNLGlCQUFpQixHQUFHLENBQUEsVUFBVSxhQUFWLFVBQVUsdUJBQVYsVUFBVSxDQUFFLGlCQUFpQixLQUFJLHVCQUF1QixJQUFJLGlCQUFpQixDQUFDLElBQUksQ0FBQztRQUM3Ryx3SEFBd0g7UUFDeEgsSUFBSSxDQUFBLFVBQVUsYUFBVixVQUFVLHVCQUFWLFVBQVUsQ0FBRSxpQkFBaUIsS0FBSSx1QkFBdUIsSUFBSSxDQUFBLFVBQVUsYUFBVixVQUFVLHVCQUFWLFVBQVUsQ0FBRSxpQkFBaUIsTUFBSyx1QkFBdUIsRUFBRTtZQUN2SCxNQUFNLElBQUksS0FBSyxDQUFDLEdBQUcsSUFBSSxDQUFDLFFBQVEsSUFBSSxJQUFJLENBQUMsVUFBVSxtQ0FBbUMsdUJBQXVCLEdBQUc7Z0JBQzVHLCtEQUErRCxVQUFVLENBQUMsaUJBQWlCLEdBQUcsQ0FBQyxDQUFDO1NBQ3ZHO1FBQ0QsSUFBSSx1QkFBVSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUNyQyxVQUFVLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztTQUN6QztRQUNELE1BQU0sV0FBVyxHQUFtQjtZQUNoQyxVQUFVLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxVQUFVO1lBQ3JDLFNBQVMsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVM7WUFDakMsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO1lBQzNCLGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYSxJQUFJLG9CQUFvQixDQUFDLGFBQWE7WUFDMUUsY0FBYyxFQUFFLE9BQU8sQ0FBQyxjQUFjLElBQUksb0JBQW9CLENBQUMsY0FBYztZQUM3RSxpQkFBaUI7WUFDakIsWUFBWTtZQUNaLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsSUFBSSxvQkFBb0IsQ0FBQyxpQkFBaUI7WUFDdEYsV0FBVyxFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDO1lBQ3RELGVBQWUsRUFBRSxJQUFJLENBQUMscUJBQXFCLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQztZQUNwRSxhQUFhLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUM7WUFDOUQsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLE9BQU8sQ0FBQztZQUNwRCxtQkFBbUIsUUFBRSxPQUFPLENBQUMsbUJBQW1CLG1DQUFJLG9CQUFvQixDQUFDLG1CQUFtQjtTQUMvRixDQUFDO1FBQ0YsTUFBTSxRQUFRLEdBQUcsSUFBSSxnQ0FBUyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDOUQsSUFBSSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQzdCLEtBQUssQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMzQyxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQztRQUMzRCxJQUFJLFVBQVUsRUFBRTtZQUNaLFVBQVUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3hDLFVBQVUsQ0FBQyxjQUFjLENBQUMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQztTQUN0RDtJQUNMLENBQUM7SUFDRDs7Ozs7Ozs7O09BU0c7SUFDSCxJQUFXLFNBQVM7UUFDaEIsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFO1lBQy9CLE1BQU0sSUFBSSxLQUFLLENBQUMsdUNBQXVDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSx1REFBdUQ7Z0JBQ3RILG9GQUFvRixDQUFDLENBQUM7U0FDN0Y7UUFDRCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDaEUsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDakcsQ0FBQztJQUNEOzs7T0FHRztJQUNILElBQVcsYUFBYTtRQUNwQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxVQUFVLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO0lBQy9HLENBQUM7SUFDTyxpQkFBaUIsQ0FBQyxXQUF5QjtRQUMvQyxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ2QsNkNBQTZDO1lBQzdDLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsRUFBRTtnQkFDbEMsT0FBTyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO2FBQ25FO1lBQ0QsbUJBQW1CO1lBQ25CLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksc0JBQWUsRUFBRSxDQUFDLENBQUM7U0FDeEQ7UUFDRCxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3ZCLE1BQU0sT0FBTyxHQUFHLFdBQVcsQ0FBQyxNQUFNLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQztRQUNqRCxJQUFJLFdBQVcsQ0FBQztRQUNoQixJQUFJLE9BQU8sQ0FBQyxzQkFBc0IsS0FBSyxTQUFTLElBQUksT0FBTyxDQUFDLGVBQWUsS0FBSyxTQUFTLEVBQUU7WUFDdkYsTUFBTSxJQUFJLEtBQUssQ0FBQywyRUFBMkUsQ0FBQyxDQUFDO1NBQ2hHO1FBQ0QsSUFBSSxPQUFPLENBQUMsY0FBYyxLQUFLLDRCQUFjLENBQUMsUUFBUSxJQUFJLE9BQU8sQ0FBQyxPQUFPLEtBQUssU0FBUyxFQUFFO1lBQ3JGLE1BQU0sSUFBSSxLQUFLLENBQUMsb0VBQW9FLENBQUMsQ0FBQztTQUN6RjtRQUNELElBQUksT0FBTyxDQUFDLGNBQWMsS0FBSyw0QkFBYyxDQUFDLFFBQVEsSUFBSSxPQUFPLENBQUMsT0FBTyxLQUFLLFNBQVMsRUFBRTtZQUNyRixNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxDQUFDLENBQUM7U0FDbEY7UUFDRCxJQUFJLE9BQU8sQ0FBQyxlQUFlLEVBQUU7WUFDekIsV0FBVyxHQUFHLE9BQU8sQ0FBQyxlQUFlLENBQUMsT0FBTyxDQUFDO1NBQ2pEO2FBQ0ksSUFBSSxPQUFPLENBQUMsc0JBQXNCLEVBQUU7WUFDckMsd0JBQXdCO1lBQ3hCLDJDQUEyQztZQUMzQyxXQUFXLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUMsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSxFQUFFLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxRQUFRLEVBQUUsTUFBTSxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsWUFBWSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDdkk7UUFDRCxPQUFPO1lBQ0gsSUFBSSxFQUFFLFdBQVcsQ0FBQyxNQUFNLENBQUMsSUFBSTtZQUM3QixHQUFHLEVBQUUsV0FBVyxDQUFDLE1BQU0sQ0FBQyxHQUFHO1lBQzNCLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxrQkFBa0I7WUFDOUMsY0FBYyxFQUFFLE9BQU8sQ0FBQyxjQUFjO1lBQ3RDLGVBQWUsRUFBRSxPQUFPLENBQUMsZUFBZTtZQUN4QyxxQkFBcUIsRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLHFCQUFxQjtZQUMvRCxpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCO1lBQzVDLGdCQUFnQixFQUFFLE9BQU8sQ0FBQyxnQkFBZ0I7WUFDMUMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLG1CQUFtQjtZQUNoRCxvQkFBb0IsRUFBRSxPQUFPLENBQUMsb0JBQW9CO1lBQ2xELGNBQWMsRUFBRSxPQUFPLENBQUMsY0FBYztZQUN0QyxZQUFZLEVBQUUsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDckUsV0FBVztTQUNkLENBQUM7SUFDTixDQUFDO0lBQ08scUJBQXFCLENBQUMsZUFBNkM7UUFDdkUsSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUNsQix1QkFBdUI7WUFDdkIsT0FBTyxTQUFTLENBQUM7U0FDcEI7UUFDRCxPQUFPLGVBQWUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUU7WUFDNUIsSUFBSSxjQUVTLENBQUM7WUFDZCxJQUFJLEVBQUUsQ0FBQyxjQUFjLEVBQUU7Z0JBQ25CLGNBQWMsR0FBRyxFQUFFLENBQUM7Z0JBQ3BCLEtBQUssTUFBTSxXQUFXLElBQUksRUFBRSxDQUFDLGNBQWMsRUFBRTtvQkFDekMsSUFBSSxFQUFFLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsRUFBRTt3QkFDL0MsY0FBYyxDQUFDLFdBQVcsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLENBQUMsT0FBTyxDQUFDO3FCQUN4RTtpQkFDSjthQUNKO1lBQ0QsTUFBTSxrQkFBa0IsR0FBRztnQkFDdkIsVUFBVSxFQUFFLEVBQUUsQ0FBQyxVQUFVO2dCQUN6QixrQkFBa0IsRUFBRSxFQUFFLENBQUMsa0JBQWtCO2dCQUN6QyxjQUFjO2FBQ2pCLENBQUM7WUFDRixPQUFPLGtCQUFrQixDQUFDO1FBQzlCLENBQUMsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNPLG1CQUFtQixDQUFDLGFBRWY7UUFHVCxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ2hCLHVCQUF1QjtZQUN2QixPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE1BQU0sTUFBTSxHQUVSLEVBQUUsQ0FBQztRQUNQLEtBQUssTUFBTSxXQUFXLElBQUksYUFBYSxFQUFFO1lBQ3JDLElBQUksYUFBYSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsRUFBRTtnQkFDM0MsTUFBTSxDQUFDLFdBQVcsQ0FBQyxHQUFHLGFBQWEsQ0FBQyxXQUFXLENBQUMsQ0FBQyxPQUFPLENBQUM7YUFDNUQ7U0FDSjtRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2xCLENBQUM7SUFDTyxrQkFBa0IsQ0FBQyxPQUFzQjs7UUFDN0MsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLElBQUksT0FBTyxDQUFDLHVCQUF1QixFQUFFO1lBQzdELE1BQU0sSUFBSSxLQUFLLENBQUMsb0ZBQW9GLENBQUMsQ0FBQztTQUN6RztRQUNELElBQUksT0FBTyxDQUFDLHVCQUF1QixFQUFFO1lBQ2pDLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsbUJBQW1CLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1lBQ2pHLE9BQU8sU0FBUyxDQUFDLGtCQUFrQixDQUFDO1NBQ3ZDO1FBQ0QsNkJBQTZCO1FBQzdCLGFBQU8sT0FBTyxDQUFDLGdCQUFnQiwwQ0FBRSxrQkFBa0IsQ0FBQztJQUN4RCxDQUFDO0NBQ0o7QUEvS0Qsd0JBK0tDO0FBQ0QsSUFBWSxpQkFpQlg7QUFqQkQsV0FBWSxpQkFBaUI7SUFDekI7O09BRUc7SUFDSCxrQ0FBYSxDQUFBO0lBQ2I7O09BRUc7SUFDSCxvQ0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCxzQ0FBaUIsQ0FBQTtJQUNqQjs7T0FFRztJQUNILG1EQUE4QixDQUFBO0FBQ2xDLENBQUMsRUFqQlcsaUJBQWlCLEdBQWpCLHlCQUFpQixLQUFqQix5QkFBaUIsUUFpQjVCO0FBQ0QsU0FBUyxVQUFVLENBQUMsSUFBWTtJQUM1QixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsMERBQTBEO0FBQ3ZHLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb25zdHJ1Y3QsIFJlc291cmNlLCBTdGFjayB9IGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQ2ZuTWV0aG9kLCBDZm5NZXRob2RQcm9wcyB9IGZyb20gJy4vYXBpZ2F0ZXdheS5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgQXV0aG9yaXplciwgSUF1dGhvcml6ZXIgfSBmcm9tICcuL2F1dGhvcml6ZXInO1xuaW1wb3J0IHsgQ29ubmVjdGlvblR5cGUsIEludGVncmF0aW9uIH0gZnJvbSAnLi9pbnRlZ3JhdGlvbic7XG5pbXBvcnQgeyBNb2NrSW50ZWdyYXRpb24gfSBmcm9tICcuL2ludGVncmF0aW9ucy9tb2NrJztcbmltcG9ydCB7IE1ldGhvZFJlc3BvbnNlIH0gZnJvbSAnLi9tZXRob2RyZXNwb25zZSc7XG5pbXBvcnQgeyBJTW9kZWwgfSBmcm9tICcuL21vZGVsJztcbmltcG9ydCB7IElSZXF1ZXN0VmFsaWRhdG9yLCBSZXF1ZXN0VmFsaWRhdG9yT3B0aW9ucyB9IGZyb20gJy4vcmVxdWVzdHZhbGlkYXRvcic7XG5pbXBvcnQgeyBJUmVzb3VyY2UgfSBmcm9tICcuL3Jlc291cmNlJztcbmltcG9ydCB7IFJlc3RBcGkgfSBmcm9tICcuL3Jlc3RhcGknO1xuaW1wb3J0IHsgdmFsaWRhdGVIdHRwTWV0aG9kIH0gZnJvbSAnLi91dGlsJztcbmV4cG9ydCBpbnRlcmZhY2UgTWV0aG9kT3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogQSBmcmllbmRseSBvcGVyYXRpb24gbmFtZSBmb3IgdGhlIG1ldGhvZC4gRm9yIGV4YW1wbGUsIHlvdSBjYW4gYXNzaWduIHRoZVxuICAgICAqIE9wZXJhdGlvbk5hbWUgb2YgTGlzdFBldHMgZm9yIHRoZSBHRVQgL3BldHMgbWV0aG9kLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG9wZXJhdGlvbk5hbWU/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogTWV0aG9kIGF1dGhvcml6YXRpb24uXG4gICAgICogSWYgdGhlIHZhbHVlIGlzIHNldCBvZiBgQ3VzdG9tYCwgYW4gYGF1dGhvcml6ZXJgIG11c3QgYWxzbyBiZSBzcGVjaWZpZWQuXG4gICAgICpcbiAgICAgKiBJZiB5b3UncmUgdXNpbmcgb25lIG9mIHRoZSBhdXRob3JpemVycyB0aGF0IGFyZSBhdmFpbGFibGUgdmlhIHRoZSB7QGxpbmsgQXV0aG9yaXplcn0gY2xhc3MsIHN1Y2ggYXMge0BsaW5rIEF1dGhvcml6ZXIjdG9rZW4oKX0sXG4gICAgICogaXQgaXMgcmVjb21tZW5kZWQgdGhhdCB0aGlzIG9wdGlvbiBub3QgYmUgc3BlY2lmaWVkLiBUaGUgYXV0aG9yaXplciB3aWxsIHRha2UgY2FyZSBvZiBzZXR0aW5nIHRoZSBjb3JyZWN0IGF1dGhvcml6YXRpb24gdHlwZS5cbiAgICAgKiBIb3dldmVyLCBzcGVjaWZ5aW5nIGFuIGF1dGhvcml6YXRpb24gdHlwZSB1c2luZyB0aGlzIHByb3BlcnR5IHRoYXQgY29uZmxpY3RzIHdpdGggd2hhdCBpcyBleHBlY3RlZCBieSB0aGUge0BsaW5rIEF1dGhvcml6ZXJ9XG4gICAgICogd2lsbCByZXN1bHQgaW4gYW4gZXJyb3IuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIG9wZW4gYWNjZXNzIHVubGVzcyBgYXV0aG9yaXplcmAgaXMgc3BlY2lmaWVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXV0aG9yaXphdGlvblR5cGU/OiBBdXRob3JpemF0aW9uVHlwZTtcbiAgICAvKipcbiAgICAgKiBJZiBgYXV0aG9yaXphdGlvblR5cGVgIGlzIGBDdXN0b21gLCB0aGlzIHNwZWNpZmllcyB0aGUgSUQgb2YgdGhlIG1ldGhvZFxuICAgICAqIGF1dGhvcml6ZXIgcmVzb3VyY2UuXG4gICAgICogSWYgc3BlY2lmaWVkLCB0aGUgdmFsdWUgb2YgYGF1dGhvcml6YXRpb25UeXBlYCBtdXN0IGJlIHNldCB0byBgQ3VzdG9tYFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGF1dGhvcml6ZXI/OiBJQXV0aG9yaXplcjtcbiAgICAvKipcbiAgICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0aGUgbWV0aG9kIHJlcXVpcmVzIGNsaWVudHMgdG8gc3VibWl0IGEgdmFsaWQgQVBJIGtleS5cbiAgICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFwaUtleVJlcXVpcmVkPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBUaGUgcmVzcG9uc2VzIHRoYXQgY2FuIGJlIHNlbnQgdG8gdGhlIGNsaWVudCB3aG8gY2FsbHMgdGhlIG1ldGhvZC5cbiAgICAgKiBAZGVmYXVsdCBOb25lXG4gICAgICpcbiAgICAgKiBUaGlzIHByb3BlcnR5IGlzIG5vdCByZXF1aXJlZCwgYnV0IGlmIHRoZXNlIGFyZSBub3Qgc3VwcGxpZWQgZm9yIGEgTGFtYmRhXG4gICAgICogcHJveHkgaW50ZWdyYXRpb24sIHRoZSBMYW1iZGEgZnVuY3Rpb24gbXVzdCByZXR1cm4gYSB2YWx1ZSBvZiB0aGUgY29ycmVjdCBmb3JtYXQsXG4gICAgICogZm9yIHRoZSBpbnRlZ3JhdGlvbiByZXNwb25zZSB0byBiZSBjb3JyZWN0bHkgbWFwcGVkIHRvIGEgcmVzcG9uc2UgdG8gdGhlIGNsaWVudC5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGktZ2F0ZXdheS1tZXRob2Qtc2V0dGluZ3MtbWV0aG9kLXJlc3BvbnNlLmh0bWxcbiAgICAgKi9cbiAgICByZWFkb25seSBtZXRob2RSZXNwb25zZXM/OiBNZXRob2RSZXNwb25zZVtdO1xuICAgIC8qKlxuICAgICAqIFRoZSByZXF1ZXN0IHBhcmFtZXRlcnMgdGhhdCBBUEkgR2F0ZXdheSBhY2NlcHRzLiBTcGVjaWZ5IHJlcXVlc3QgcGFyYW1ldGVyc1xuICAgICAqIGFzIGtleS12YWx1ZSBwYWlycyAoc3RyaW5nLXRvLUJvb2xlYW4gbWFwcGluZyksIHdpdGggYSBzb3VyY2UgYXMgdGhlIGtleSBhbmRcbiAgICAgKiBhIEJvb2xlYW4gYXMgdGhlIHZhbHVlLiBUaGUgQm9vbGVhbiBzcGVjaWZpZXMgd2hldGhlciBhIHBhcmFtZXRlciBpcyByZXF1aXJlZC5cbiAgICAgKiBBIHNvdXJjZSBtdXN0IG1hdGNoIHRoZSBmb3JtYXQgbWV0aG9kLnJlcXVlc3QubG9jYXRpb24ubmFtZSwgd2hlcmUgdGhlIGxvY2F0aW9uXG4gICAgICogaXMgcXVlcnlzdHJpbmcsIHBhdGgsIG9yIGhlYWRlciwgYW5kIG5hbWUgaXMgYSB2YWxpZCwgdW5pcXVlIHBhcmFtZXRlciBuYW1lLlxuICAgICAqIEBkZWZhdWx0IE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSByZXF1ZXN0UGFyYW1ldGVycz86IHtcbiAgICAgICAgW3BhcmFtOiBzdHJpbmddOiBib29sZWFuO1xuICAgIH07XG4gICAgLyoqXG4gICAgICogVGhlIG1vZGVscyB3aGljaCBkZXNjcmliZSBkYXRhIHN0cnVjdHVyZSBvZiByZXF1ZXN0IHBheWxvYWQuIFdoZW5cbiAgICAgKiBjb21iaW5lZCB3aXRoIGByZXF1ZXN0VmFsaWRhdG9yYCBvciBgcmVxdWVzdFZhbGlkYXRvck9wdGlvbnNgLCB0aGUgc2VydmljZVxuICAgICAqIHdpbGwgdmFsaWRhdGUgdGhlIEFQSSByZXF1ZXN0IHBheWxvYWQgYmVmb3JlIGl0IHJlYWNoZXMgdGhlIEFQSSdzIEludGVncmF0aW9uIChpbmNsdWRpbmcgcHJveGllcykuXG4gICAgICogU3BlY2lmeSBgcmVxdWVzdE1vZGVsc2AgYXMga2V5LXZhbHVlIHBhaXJzLCB3aXRoIGEgY29udGVudCB0eXBlXG4gICAgICogKGUuZy4gYCdhcHBsaWNhdGlvbi9qc29uJ2ApIGFzIHRoZSBrZXkgYW5kIGFuIEFQSSBHYXRld2F5IE1vZGVsIGFzIHRoZSB2YWx1ZS5cbiAgICAgKlxuICAgICAqIEBleGFtcGxlXG4gICAgICpcbiAgICAgKiAgICAgY29uc3QgdXNlck1vZGVsOiBhcGlnYXRld2F5Lk1vZGVsID0gYXBpLmFkZE1vZGVsKCdVc2VyTW9kZWwnLCB7XG4gICAgICogICAgICAgICBzY2hlbWE6IHtcbiAgICAgKiAgICAgICAgICAgICB0eXBlOiBhcGlnYXRld2F5Lkpzb25TY2hlbWFUeXBlLk9CSkVDVFxuICAgICAqICAgICAgICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgKiAgICAgICAgICAgICAgICAgdXNlcklkOiB7XG4gICAgICogICAgICAgICAgICAgICAgICAgICB0eXBlOiBhcGlnYXRld2F5Lkpzb25TY2hlbWEuU1RSSU5HXG4gICAgICogICAgICAgICAgICAgICAgIH0sXG4gICAgICogICAgICAgICAgICAgICAgIG5hbWU6IHtcbiAgICAgKiAgICAgICAgICAgICAgICAgICAgIHR5cGU6IGFwaWdhdGV3YXkuSnNvblNjaGVtYS5TVFJJTkdcbiAgICAgKiAgICAgICAgICAgICAgICAgfVxuICAgICAqICAgICAgICAgICAgIH0sXG4gICAgICogICAgICAgICAgICAgcmVxdWlyZWQ6IFsndXNlcklkJ11cbiAgICAgKiAgICAgICAgIH1cbiAgICAgKiAgICAgfSk7XG4gICAgICogICAgIGFwaS5yb290LmFkZFJlc291cmNlKCd1c2VyJykuYWRkTWV0aG9kKCdQT1NUJyxcbiAgICAgKiAgICAgICAgIG5ldyBhcGlnYXRld2F5LkxhbWJkYUludGVncmF0aW9uKHVzZXJMYW1iZGEpLCB7XG4gICAgICogICAgICAgICAgICAgcmVxdWVzdE1vZGVsczoge1xuICAgICAqICAgICAgICAgICAgICAgICAnYXBwbGljYXRpb24vanNvbic6IHVzZXJNb2RlbFxuICAgICAqICAgICAgICAgICAgIH1cbiAgICAgKiAgICAgICAgIH1cbiAgICAgKiAgICAgKTtcbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FwaS1nYXRld2F5LW1ldGhvZC1zZXR0aW5ncy1tZXRob2QtcmVxdWVzdC5odG1sI3NldHVwLW1ldGhvZC1yZXF1ZXN0LW1vZGVsXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVxdWVzdE1vZGVscz86IHtcbiAgICAgICAgW3BhcmFtOiBzdHJpbmddOiBJTW9kZWw7XG4gICAgfTtcbiAgICAvKipcbiAgICAgKiBUaGUgSUQgb2YgdGhlIGFzc29jaWF0ZWQgcmVxdWVzdCB2YWxpZGF0b3IuXG4gICAgICogT25seSBvbmUgb2YgYHJlcXVlc3RWYWxpZGF0b3JgIG9yIGByZXF1ZXN0VmFsaWRhdG9yT3B0aW9uc2AgbXVzdCBiZSBzcGVjaWZpZWQuXG4gICAgICogV29ya3MgdG9nZXRoZXIgd2l0aCBgcmVxdWVzdE1vZGVsc2Agb3IgYHJlcXVlc3RQYXJhbWV0ZXJzYCB0byB2YWxpZGF0ZVxuICAgICAqIHRoZSByZXF1ZXN0IGJlZm9yZSBpdCByZWFjaGVzIGludGVncmF0aW9uIGxpa2UgTGFtYmRhIFByb3h5IEludGVncmF0aW9uLlxuICAgICAqIEBkZWZhdWx0IC0gTm8gZGVmYXVsdCB2YWxpZGF0b3JcbiAgICAgKi9cbiAgICByZWFkb25seSByZXF1ZXN0VmFsaWRhdG9yPzogSVJlcXVlc3RWYWxpZGF0b3I7XG4gICAgLyoqXG4gICAgICogQSBsaXN0IG9mIGF1dGhvcml6YXRpb24gc2NvcGVzIGNvbmZpZ3VyZWQgb24gdGhlIG1ldGhvZC4gVGhlIHNjb3BlcyBhcmUgdXNlZCB3aXRoXG4gICAgICogYSBDT0dOSVRPX1VTRVJfUE9PTFMgYXV0aG9yaXplciB0byBhdXRob3JpemUgdGhlIG1ldGhvZCBpbnZvY2F0aW9uLlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWFwaWdhdGV3YXktbWV0aG9kLmh0bWwjY2ZuLWFwaWdhdGV3YXktbWV0aG9kLWF1dGhvcml6YXRpb25zY29wZXNcbiAgICAgKiBAZGVmYXVsdCAtIG5vIGF1dGhvcml6YXRpb24gc2NvcGVzXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXV0aG9yaXphdGlvblNjb3Blcz86IHN0cmluZ1tdO1xuICAgIC8qKlxuICAgICAqIFJlcXVlc3QgdmFsaWRhdG9yIG9wdGlvbnMgdG8gY3JlYXRlIG5ldyB2YWxpZGF0b3JcbiAgICAgKiBPbmx5IG9uZSBvZiBgcmVxdWVzdFZhbGlkYXRvcmAgb3IgYHJlcXVlc3RWYWxpZGF0b3JPcHRpb25zYCBtdXN0IGJlIHNwZWNpZmllZC5cbiAgICAgKiBXb3JrcyB0b2dldGhlciB3aXRoIGByZXF1ZXN0TW9kZWxzYCBvciBgcmVxdWVzdFBhcmFtZXRlcnNgIHRvIHZhbGlkYXRlXG4gICAgICogdGhlIHJlcXVlc3QgYmVmb3JlIGl0IHJlYWNoZXMgaW50ZWdyYXRpb24gbGlrZSBMYW1iZGEgUHJveHkgSW50ZWdyYXRpb24uXG4gICAgICogQGRlZmF1bHQgLSBObyBkZWZhdWx0IHZhbGlkYXRvclxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlcXVlc3RWYWxpZGF0b3JPcHRpb25zPzogUmVxdWVzdFZhbGlkYXRvck9wdGlvbnM7XG59XG5leHBvcnQgaW50ZXJmYWNlIE1ldGhvZFByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgcmVzb3VyY2UgdGhpcyBtZXRob2QgaXMgYXNzb2NpYXRlZCB3aXRoLiBGb3Igcm9vdCByZXNvdXJjZSBtZXRob2RzLFxuICAgICAqIHNwZWNpZnkgdGhlIGBSZXN0QXBpYCBvYmplY3QuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVzb3VyY2U6IElSZXNvdXJjZTtcbiAgICAvKipcbiAgICAgKiBUaGUgSFRUUCBtZXRob2QgKFwiR0VUXCIsIFwiUE9TVFwiLCBcIlBVVFwiLCAuLi4pIHRoYXQgY2xpZW50cyB1c2UgdG8gY2FsbCB0aGlzIG1ldGhvZC5cbiAgICAgKi9cbiAgICByZWFkb25seSBodHRwTWV0aG9kOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGJhY2tlbmQgc3lzdGVtIHRoYXQgdGhlIG1ldGhvZCBjYWxscyB3aGVuIGl0IHJlY2VpdmVzIGEgcmVxdWVzdC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gYSBuZXcgYE1vY2tJbnRlZ3JhdGlvbmAuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbjtcbiAgICAvKipcbiAgICAgKiBNZXRob2Qgb3B0aW9ucy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gb3B0aW9ucy5cbiAgICAgKi9cbiAgICByZWFkb25seSBvcHRpb25zPzogTWV0aG9kT3B0aW9ucztcbn1cbmV4cG9ydCBjbGFzcyBNZXRob2QgZXh0ZW5kcyBSZXNvdXJjZSB7XG4gICAgLyoqIEBhdHRyaWJ1dGUgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgbWV0aG9kSWQ6IHN0cmluZztcbiAgICBwdWJsaWMgcmVhZG9ubHkgaHR0cE1ldGhvZDogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSByZXNvdXJjZTogSVJlc291cmNlO1xuICAgIHB1YmxpYyByZWFkb25seSByZXN0QXBpOiBSZXN0QXBpO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBNZXRob2RQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgICAgICB0aGlzLnJlc291cmNlID0gcHJvcHMucmVzb3VyY2U7XG4gICAgICAgIHRoaXMucmVzdEFwaSA9IHByb3BzLnJlc291cmNlLnJlc3RBcGk7XG4gICAgICAgIHRoaXMuaHR0cE1ldGhvZCA9IHByb3BzLmh0dHBNZXRob2QudG9VcHBlckNhc2UoKTtcbiAgICAgICAgdmFsaWRhdGVIdHRwTWV0aG9kKHRoaXMuaHR0cE1ldGhvZCk7XG4gICAgICAgIGNvbnN0IG9wdGlvbnMgPSBwcm9wcy5vcHRpb25zIHx8IHt9O1xuICAgICAgICBjb25zdCBkZWZhdWx0TWV0aG9kT3B0aW9ucyA9IHByb3BzLnJlc291cmNlLmRlZmF1bHRNZXRob2RPcHRpb25zIHx8IHt9O1xuICAgICAgICBjb25zdCBhdXRob3JpemVyID0gb3B0aW9ucy5hdXRob3JpemVyIHx8IGRlZmF1bHRNZXRob2RPcHRpb25zLmF1dGhvcml6ZXI7XG4gICAgICAgIGNvbnN0IGF1dGhvcml6ZXJJZCA9IGF1dGhvcml6ZXI/LmF1dGhvcml6ZXJJZDtcbiAgICAgICAgY29uc3QgYXV0aG9yaXphdGlvblR5cGVPcHRpb24gPSBvcHRpb25zLmF1dGhvcml6YXRpb25UeXBlIHx8IGRlZmF1bHRNZXRob2RPcHRpb25zLmF1dGhvcml6YXRpb25UeXBlO1xuICAgICAgICBjb25zdCBhdXRob3JpemF0aW9uVHlwZSA9IGF1dGhvcml6ZXI/LmF1dGhvcml6YXRpb25UeXBlIHx8IGF1dGhvcml6YXRpb25UeXBlT3B0aW9uIHx8IEF1dGhvcml6YXRpb25UeXBlLk5PTkU7XG4gICAgICAgIC8vIGlmIHRoZSBhdXRob3JpemVyIGRlZmluZXMgYW4gYXV0aG9yaXphdGlvbiB0eXBlIGFuZCB3ZSBhbHNvIGhhdmUgYW4gZXhwbGljaXQgb3B0aW9uIHNldCwgY2hlY2sgdGhhdCB0aGV5IGFyZSB0aGUgc2FtZVxuICAgICAgICBpZiAoYXV0aG9yaXplcj8uYXV0aG9yaXphdGlvblR5cGUgJiYgYXV0aG9yaXphdGlvblR5cGVPcHRpb24gJiYgYXV0aG9yaXplcj8uYXV0aG9yaXphdGlvblR5cGUgIT09IGF1dGhvcml6YXRpb25UeXBlT3B0aW9uKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYCR7dGhpcy5yZXNvdXJjZX0vJHt0aGlzLmh0dHBNZXRob2R9IC0gQXV0aG9yaXphdGlvbiB0eXBlIGlzIHNldCB0byAke2F1dGhvcml6YXRpb25UeXBlT3B0aW9ufSBgICtcbiAgICAgICAgICAgICAgICBgd2hpY2ggaXMgZGlmZmVyZW50IGZyb20gd2hhdCBpcyByZXF1aXJlZCBieSB0aGUgYXV0aG9yaXplciBbJHthdXRob3JpemVyLmF1dGhvcml6YXRpb25UeXBlfV1gKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoQXV0aG9yaXplci5pc0F1dGhvcml6ZXIoYXV0aG9yaXplcikpIHtcbiAgICAgICAgICAgIGF1dGhvcml6ZXIuX2F0dGFjaFRvQXBpKHRoaXMucmVzdEFwaSk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgbWV0aG9kUHJvcHM6IENmbk1ldGhvZFByb3BzID0ge1xuICAgICAgICAgICAgcmVzb3VyY2VJZDogcHJvcHMucmVzb3VyY2UucmVzb3VyY2VJZCxcbiAgICAgICAgICAgIHJlc3RBcGlJZDogdGhpcy5yZXN0QXBpLnJlc3RBcGlJZCxcbiAgICAgICAgICAgIGh0dHBNZXRob2Q6IHRoaXMuaHR0cE1ldGhvZCxcbiAgICAgICAgICAgIG9wZXJhdGlvbk5hbWU6IG9wdGlvbnMub3BlcmF0aW9uTmFtZSB8fCBkZWZhdWx0TWV0aG9kT3B0aW9ucy5vcGVyYXRpb25OYW1lLFxuICAgICAgICAgICAgYXBpS2V5UmVxdWlyZWQ6IG9wdGlvbnMuYXBpS2V5UmVxdWlyZWQgfHwgZGVmYXVsdE1ldGhvZE9wdGlvbnMuYXBpS2V5UmVxdWlyZWQsXG4gICAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZSxcbiAgICAgICAgICAgIGF1dGhvcml6ZXJJZCxcbiAgICAgICAgICAgIHJlcXVlc3RQYXJhbWV0ZXJzOiBvcHRpb25zLnJlcXVlc3RQYXJhbWV0ZXJzIHx8IGRlZmF1bHRNZXRob2RPcHRpb25zLnJlcXVlc3RQYXJhbWV0ZXJzLFxuICAgICAgICAgICAgaW50ZWdyYXRpb246IHRoaXMucmVuZGVySW50ZWdyYXRpb24ocHJvcHMuaW50ZWdyYXRpb24pLFxuICAgICAgICAgICAgbWV0aG9kUmVzcG9uc2VzOiB0aGlzLnJlbmRlck1ldGhvZFJlc3BvbnNlcyhvcHRpb25zLm1ldGhvZFJlc3BvbnNlcyksXG4gICAgICAgICAgICByZXF1ZXN0TW9kZWxzOiB0aGlzLnJlbmRlclJlcXVlc3RNb2RlbHMob3B0aW9ucy5yZXF1ZXN0TW9kZWxzKSxcbiAgICAgICAgICAgIHJlcXVlc3RWYWxpZGF0b3JJZDogdGhpcy5yZXF1ZXN0VmFsaWRhdG9ySWQob3B0aW9ucyksXG4gICAgICAgICAgICBhdXRob3JpemF0aW9uU2NvcGVzOiBvcHRpb25zLmF1dGhvcml6YXRpb25TY29wZXMgPz8gZGVmYXVsdE1ldGhvZE9wdGlvbnMuYXV0aG9yaXphdGlvblNjb3BlcyxcbiAgICAgICAgfTtcbiAgICAgICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuTWV0aG9kKHRoaXMsICdSZXNvdXJjZScsIG1ldGhvZFByb3BzKTtcbiAgICAgICAgdGhpcy5tZXRob2RJZCA9IHJlc291cmNlLnJlZjtcbiAgICAgICAgcHJvcHMucmVzb3VyY2UucmVzdEFwaS5fYXR0YWNoTWV0aG9kKHRoaXMpO1xuICAgICAgICBjb25zdCBkZXBsb3ltZW50ID0gcHJvcHMucmVzb3VyY2UucmVzdEFwaS5sYXRlc3REZXBsb3ltZW50O1xuICAgICAgICBpZiAoZGVwbG95bWVudCkge1xuICAgICAgICAgICAgZGVwbG95bWVudC5ub2RlLmFkZERlcGVuZGVuY3kocmVzb3VyY2UpO1xuICAgICAgICAgICAgZGVwbG95bWVudC5hZGRUb0xvZ2ljYWxJZCh7IG1ldGhvZDogbWV0aG9kUHJvcHMgfSk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJucyBhbiBleGVjdXRlLWFwaSBBUk4gZm9yIHRoaXMgbWV0aG9kOlxuICAgICAqXG4gICAgICogICBhcm46YXdzOmV4ZWN1dGUtYXBpOntyZWdpb259OnthY2NvdW50fTp7cmVzdEFwaUlkfS97c3RhZ2V9L3ttZXRob2R9L3twYXRofVxuICAgICAqXG4gICAgICogTk9URToge3N0YWdlfSB3aWxsIHJlZmVyIHRvIHRoZSBgcmVzdEFwaS5kZXBsb3ltZW50U3RhZ2VgLCB3aGljaCB3aWxsXG4gICAgICogYXV0b21hdGljYWxseSBzZXQgaWYgYXV0by1kZXBsb3kgaXMgZW5hYmxlZC5cbiAgICAgKlxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICBwdWJsaWMgZ2V0IG1ldGhvZEFybigpOiBzdHJpbmcge1xuICAgICAgICBpZiAoIXRoaXMucmVzdEFwaS5kZXBsb3ltZW50U3RhZ2UpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5hYmxlIHRvIGRldGVybWluZSBBUk4gZm9yIG1ldGhvZCBcIiR7dGhpcy5ub2RlLmlkfVwiIHNpbmNlIHRoZXJlIGlzIG5vIHN0YWdlIGFzc29jaWF0ZWQgd2l0aCB0aGlzIEFQSS5cXG5gICtcbiAgICAgICAgICAgICAgICAnRWl0aGVyIHVzZSB0aGUgYGRlcGxveWAgcHJvcCBvciBleHBsaWNpdGx5IGFzc2lnbiBgZGVwbG95bWVudFN0YWdlYCBvbiB0aGUgUmVzdEFwaScpO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHN0YWdlID0gdGhpcy5yZXN0QXBpLmRlcGxveW1lbnRTdGFnZS5zdGFnZU5hbWUudG9TdHJpbmcoKTtcbiAgICAgICAgcmV0dXJuIHRoaXMucmVzdEFwaS5hcm5Gb3JFeGVjdXRlQXBpKHRoaXMuaHR0cE1ldGhvZCwgcGF0aEZvckFybih0aGlzLnJlc291cmNlLnBhdGgpLCBzdGFnZSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHVybnMgYW4gZXhlY3V0ZS1hcGkgQVJOIGZvciB0aGlzIG1ldGhvZCdzIFwidGVzdC1pbnZva2Utc3RhZ2VcIiBzdGFnZS5cbiAgICAgKiBUaGlzIHN0YWdlIGlzIHVzZWQgYnkgdGhlIEFXUyBDb25zb2xlIFVJIHdoZW4gdGVzdGluZyB0aGUgbWV0aG9kLlxuICAgICAqL1xuICAgIHB1YmxpYyBnZXQgdGVzdE1ldGhvZEFybigpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gdGhpcy5yZXN0QXBpLmFybkZvckV4ZWN1dGVBcGkodGhpcy5odHRwTWV0aG9kLCBwYXRoRm9yQXJuKHRoaXMucmVzb3VyY2UucGF0aCksICd0ZXN0LWludm9rZS1zdGFnZScpO1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlckludGVncmF0aW9uKGludGVncmF0aW9uPzogSW50ZWdyYXRpb24pOiBDZm5NZXRob2QuSW50ZWdyYXRpb25Qcm9wZXJ0eSB7XG4gICAgICAgIGlmICghaW50ZWdyYXRpb24pIHtcbiAgICAgICAgICAgIC8vIHVzZSBkZWZhdWx0SW50ZWdyYXRpb24gZnJvbSBBUEkgaWYgZGVmaW5lZFxuICAgICAgICAgICAgaWYgKHRoaXMucmVzb3VyY2UuZGVmYXVsdEludGVncmF0aW9uKSB7XG4gICAgICAgICAgICAgICAgcmV0dXJuIHRoaXMucmVuZGVySW50ZWdyYXRpb24odGhpcy5yZXNvdXJjZS5kZWZhdWx0SW50ZWdyYXRpb24pO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgLy8gZmFsbGJhY2sgdG8gbW9ja1xuICAgICAgICAgICAgcmV0dXJuIHRoaXMucmVuZGVySW50ZWdyYXRpb24obmV3IE1vY2tJbnRlZ3JhdGlvbigpKTtcbiAgICAgICAgfVxuICAgICAgICBpbnRlZ3JhdGlvbi5iaW5kKHRoaXMpO1xuICAgICAgICBjb25zdCBvcHRpb25zID0gaW50ZWdyYXRpb24uX3Byb3BzLm9wdGlvbnMgfHwge307XG4gICAgICAgIGxldCBjcmVkZW50aWFscztcbiAgICAgICAgaWYgKG9wdGlvbnMuY3JlZGVudGlhbHNQYXNzdGhyb3VnaCAhPT0gdW5kZWZpbmVkICYmIG9wdGlvbnMuY3JlZGVudGlhbHNSb2xlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignXFwnY3JlZGVudGlhbHNQYXNzdGhyb3VnaFxcJyBhbmQgXFwnY3JlZGVudGlhbHNSb2xlXFwnIGFyZSBtdXR1YWxseSBleGNsdXNpdmUnKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAob3B0aW9ucy5jb25uZWN0aW9uVHlwZSA9PT0gQ29ubmVjdGlvblR5cGUuVlBDX0xJTksgJiYgb3B0aW9ucy52cGNMaW5rID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignXFwnY29ubmVjdGlvblR5cGVcXCcgb2YgVlBDX0xJTksgcmVxdWlyZXMgXFwndnBjTGlua1xcJyBwcm9wIHRvIGJlIHNldCcpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChvcHRpb25zLmNvbm5lY3Rpb25UeXBlID09PSBDb25uZWN0aW9uVHlwZS5JTlRFUk5FVCAmJiBvcHRpb25zLnZwY0xpbmsgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdjYW5ub3Qgc2V0IFxcJ3ZwY0xpbmtcXCcgd2hlcmUgXFwnY29ubmVjdGlvblR5cGVcXCcgaXMgSU5URVJORVQnKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAob3B0aW9ucy5jcmVkZW50aWFsc1JvbGUpIHtcbiAgICAgICAgICAgIGNyZWRlbnRpYWxzID0gb3B0aW9ucy5jcmVkZW50aWFsc1JvbGUucm9sZUFybjtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIGlmIChvcHRpb25zLmNyZWRlbnRpYWxzUGFzc3Rocm91Z2gpIHtcbiAgICAgICAgICAgIC8vIGFybjphd3M6aWFtOjoqOnVzZXIvKlxuICAgICAgICAgICAgLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm1heC1saW5lLWxlbmd0aFxuICAgICAgICAgICAgY3JlZGVudGlhbHMgPSBTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oeyBzZXJ2aWNlOiAnaWFtJywgcmVnaW9uOiAnJywgYWNjb3VudDogJyonLCByZXNvdXJjZTogJ3VzZXInLCBzZXA6ICcvJywgcmVzb3VyY2VOYW1lOiAnKicgfSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIHR5cGU6IGludGVncmF0aW9uLl9wcm9wcy50eXBlLFxuICAgICAgICAgICAgdXJpOiBpbnRlZ3JhdGlvbi5fcHJvcHMudXJpLFxuICAgICAgICAgICAgY2FjaGVLZXlQYXJhbWV0ZXJzOiBvcHRpb25zLmNhY2hlS2V5UGFyYW1ldGVycyxcbiAgICAgICAgICAgIGNhY2hlTmFtZXNwYWNlOiBvcHRpb25zLmNhY2hlTmFtZXNwYWNlLFxuICAgICAgICAgICAgY29udGVudEhhbmRsaW5nOiBvcHRpb25zLmNvbnRlbnRIYW5kbGluZyxcbiAgICAgICAgICAgIGludGVncmF0aW9uSHR0cE1ldGhvZDogaW50ZWdyYXRpb24uX3Byb3BzLmludGVncmF0aW9uSHR0cE1ldGhvZCxcbiAgICAgICAgICAgIHJlcXVlc3RQYXJhbWV0ZXJzOiBvcHRpb25zLnJlcXVlc3RQYXJhbWV0ZXJzLFxuICAgICAgICAgICAgcmVxdWVzdFRlbXBsYXRlczogb3B0aW9ucy5yZXF1ZXN0VGVtcGxhdGVzLFxuICAgICAgICAgICAgcGFzc3Rocm91Z2hCZWhhdmlvcjogb3B0aW9ucy5wYXNzdGhyb3VnaEJlaGF2aW9yLFxuICAgICAgICAgICAgaW50ZWdyYXRpb25SZXNwb25zZXM6IG9wdGlvbnMuaW50ZWdyYXRpb25SZXNwb25zZXMsXG4gICAgICAgICAgICBjb25uZWN0aW9uVHlwZTogb3B0aW9ucy5jb25uZWN0aW9uVHlwZSxcbiAgICAgICAgICAgIGNvbm5lY3Rpb25JZDogb3B0aW9ucy52cGNMaW5rID8gb3B0aW9ucy52cGNMaW5rLnZwY0xpbmtJZCA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIGNyZWRlbnRpYWxzLFxuICAgICAgICB9O1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlck1ldGhvZFJlc3BvbnNlcyhtZXRob2RSZXNwb25zZXM6IE1ldGhvZFJlc3BvbnNlW10gfCB1bmRlZmluZWQpOiBDZm5NZXRob2QuTWV0aG9kUmVzcG9uc2VQcm9wZXJ0eVtdIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgaWYgKCFtZXRob2RSZXNwb25zZXMpIHtcbiAgICAgICAgICAgIC8vIEZhbGwgYmFjayB0byBub3RoaW5nXG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBtZXRob2RSZXNwb25zZXMubWFwKG1yID0+IHtcbiAgICAgICAgICAgIGxldCByZXNwb25zZU1vZGVsczoge1xuICAgICAgICAgICAgICAgIFtjb250ZW50VHlwZTogc3RyaW5nXTogc3RyaW5nO1xuICAgICAgICAgICAgfSB8IHVuZGVmaW5lZDtcbiAgICAgICAgICAgIGlmIChtci5yZXNwb25zZU1vZGVscykge1xuICAgICAgICAgICAgICAgIHJlc3BvbnNlTW9kZWxzID0ge307XG4gICAgICAgICAgICAgICAgZm9yIChjb25zdCBjb250ZW50VHlwZSBpbiBtci5yZXNwb25zZU1vZGVscykge1xuICAgICAgICAgICAgICAgICAgICBpZiAobXIucmVzcG9uc2VNb2RlbHMuaGFzT3duUHJvcGVydHkoY29udGVudFR5cGUpKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICByZXNwb25zZU1vZGVsc1tjb250ZW50VHlwZV0gPSBtci5yZXNwb25zZU1vZGVsc1tjb250ZW50VHlwZV0ubW9kZWxJZDtcbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGNvbnN0IG1ldGhvZFJlc3BvbnNlUHJvcCA9IHtcbiAgICAgICAgICAgICAgICBzdGF0dXNDb2RlOiBtci5zdGF0dXNDb2RlLFxuICAgICAgICAgICAgICAgIHJlc3BvbnNlUGFyYW1ldGVyczogbXIucmVzcG9uc2VQYXJhbWV0ZXJzLFxuICAgICAgICAgICAgICAgIHJlc3BvbnNlTW9kZWxzLFxuICAgICAgICAgICAgfTtcbiAgICAgICAgICAgIHJldHVybiBtZXRob2RSZXNwb25zZVByb3A7XG4gICAgICAgIH0pO1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlclJlcXVlc3RNb2RlbHMocmVxdWVzdE1vZGVsczoge1xuICAgICAgICBbcGFyYW06IHN0cmluZ106IElNb2RlbDtcbiAgICB9IHwgdW5kZWZpbmVkKToge1xuICAgICAgICBbcGFyYW06IHN0cmluZ106IHN0cmluZztcbiAgICB9IHwgdW5kZWZpbmVkIHtcbiAgICAgICAgaWYgKCFyZXF1ZXN0TW9kZWxzKSB7XG4gICAgICAgICAgICAvLyBGYWxsIGJhY2sgdG8gbm90aGluZ1xuICAgICAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBtb2RlbHM6IHtcbiAgICAgICAgICAgIFtwYXJhbTogc3RyaW5nXTogc3RyaW5nO1xuICAgICAgICB9ID0ge307XG4gICAgICAgIGZvciAoY29uc3QgY29udGVudFR5cGUgaW4gcmVxdWVzdE1vZGVscykge1xuICAgICAgICAgICAgaWYgKHJlcXVlc3RNb2RlbHMuaGFzT3duUHJvcGVydHkoY29udGVudFR5cGUpKSB7XG4gICAgICAgICAgICAgICAgbW9kZWxzW2NvbnRlbnRUeXBlXSA9IHJlcXVlc3RNb2RlbHNbY29udGVudFR5cGVdLm1vZGVsSWQ7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG1vZGVscztcbiAgICB9XG4gICAgcHJpdmF0ZSByZXF1ZXN0VmFsaWRhdG9ySWQob3B0aW9uczogTWV0aG9kT3B0aW9ucyk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGlmIChvcHRpb25zLnJlcXVlc3RWYWxpZGF0b3IgJiYgb3B0aW9ucy5yZXF1ZXN0VmFsaWRhdG9yT3B0aW9ucykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdPbmx5IG9uZSBvZiBcXCdyZXF1ZXN0VmFsaWRhdG9yXFwnIG9yIFxcJ3JlcXVlc3RWYWxpZGF0b3JPcHRpb25zXFwnIG11c3QgYmUgc3BlY2lmaWVkLicpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChvcHRpb25zLnJlcXVlc3RWYWxpZGF0b3JPcHRpb25zKSB7XG4gICAgICAgICAgICBjb25zdCB2YWxpZGF0b3IgPSB0aGlzLnJlc3RBcGkuYWRkUmVxdWVzdFZhbGlkYXRvcigndmFsaWRhdG9yJywgb3B0aW9ucy5yZXF1ZXN0VmFsaWRhdG9yT3B0aW9ucyk7XG4gICAgICAgICAgICByZXR1cm4gdmFsaWRhdG9yLnJlcXVlc3RWYWxpZGF0b3JJZDtcbiAgICAgICAgfVxuICAgICAgICAvLyBGb3IgYmFja3dhcmQgY29tcGF0aWJpbGl0eVxuICAgICAgICByZXR1cm4gb3B0aW9ucy5yZXF1ZXN0VmFsaWRhdG9yPy5yZXF1ZXN0VmFsaWRhdG9ySWQ7XG4gICAgfVxufVxuZXhwb3J0IGVudW0gQXV0aG9yaXphdGlvblR5cGUge1xuICAgIC8qKlxuICAgICAqIE9wZW4gYWNjZXNzLlxuICAgICAqL1xuICAgIE5PTkUgPSAnTk9ORScsXG4gICAgLyoqXG4gICAgICogVXNlIEFXUyBJQU0gcGVybWlzc2lvbnMuXG4gICAgICovXG4gICAgSUFNID0gJ0FXU19JQU0nLFxuICAgIC8qKlxuICAgICAqIFVzZSBhIGN1c3RvbSBhdXRob3JpemVyLlxuICAgICAqL1xuICAgIENVU1RPTSA9ICdDVVNUT00nLFxuICAgIC8qKlxuICAgICAqIFVzZSBhbiBBV1MgQ29nbml0byB1c2VyIHBvb2wuXG4gICAgICovXG4gICAgQ09HTklUTyA9ICdDT0dOSVRPX1VTRVJfUE9PTFMnXG59XG5mdW5jdGlvbiBwYXRoRm9yQXJuKHBhdGg6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHBhdGgucmVwbGFjZSgvXFx7W15cXH1dKlxcfS9nLCAnKicpOyAvLyByZXBsYWNlIHBhdGggcGFyYW1ldGVycyAobGlrZSAne2Jvb2tJZH0nKSB3aXRoIGFzdGVyaXNrXG59XG4iXX0=