"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const crypto = require("crypto");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 */
class Deployment extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApi: props.api,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's
     * logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Hook into synthesis before it occurs and make any final adjustments.
     */
    prepare() {
        if (this.api instanceof restapi_1.RestApi) {
            // Ignore IRestApi that are imported
            /*
             * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
             * Quoting from CloudFormation's docs - "If you create an AWS::ApiGateway::RestApi resource and its methods (using AWS::ApiGateway::Method) in
             * the same template as your deployment, the deployment must depend on the RestApi's methods. To create a dependency, add a DependsOn attribute
             * to the deployment. If you don't, AWS CloudFormation creates the deployment right after it creates the RestApi resource that doesn't contain
             * any methods, and AWS CloudFormation encounters the following error: The REST API doesn't contain any methods."
             */
            /*
             * Adding a dependency between LatestDeployment and Method construct, using ConstructNode.addDependencies(), creates additional dependencies
             * between AWS::ApiGateway::Deployment and the AWS::Lambda::Permission nodes (children under Method), causing cyclic dependency errors. Hence,
             * falling back to declaring dependencies between the underlying CfnResources.
             */
            this.api.methods.map(m => m.node.defaultChild).forEach(m => this.resource.addDependsOn(m));
        }
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, {
            description: props.description,
            restApiId: props.restApi.restApiId,
        });
        this.hashComponents = new Array();
        this.api = props.restApi;
        this.originalLogicalId = core_1.Stack.of(this).getLogicalId(this);
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    /**
     * Hooks into synthesis to calculate a logical ID that hashes all the components
     * add via `addToLogicalId`.
     */
    prepare() {
        if (this.api instanceof restapi_1.RestApi || this.api instanceof restapi_1.SpecRestApi) { // Ignore IRestApi that are imported
            // Add CfnRestApi to the logical id so a new deployment is triggered when any of its properties change.
            const cfnRestApiCF = this.api.node.defaultChild._toCloudFormation();
            this.addToLogicalId(core_1.Stack.of(this).resolve(cfnRestApiCF));
        }
        const stack = core_1.Stack.of(this);
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (this.hashComponents.length > 0) {
            const md5 = crypto.createHash('md5');
            this.hashComponents.map(x => stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            this.overrideLogicalId(this.originalLogicalId + md5.digest('hex'));
        }
        super.prepare();
    }
}
//# sourceMappingURL=data:application/json;base64,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