"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("../apigateway.generated");
const authorizer_1 = require("../authorizer");
class LambdaAuthorizer extends authorizer_1.Authorizer {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.handler = props.handler;
        this.role = props.assumeRole;
        if (props.resultsCacheTtl && ((_a = props.resultsCacheTtl) === null || _a === void 0 ? void 0 : _a.toSeconds()) > 3600) {
            throw new Error('Lambda authorizer property \'resultsCacheTtl\' must not be greater than 3600 seconds (1 hour)');
        }
    }
    /**
     * Attaches this authorizer to a specific REST API.
     * @internal
     */
    _attachToApi(restApi) {
        if (this.restApiId && this.restApiId !== restApi.restApiId) {
            throw new Error('Cannot attach authorizer to two different rest APIs');
        }
        this.restApiId = restApi.restApiId;
    }
    /**
     * Sets up the permissions necessary for the API Gateway service to invoke the Lambda function.
     */
    setupPermissions() {
        if (!this.role) {
            this.handler.addPermission(`${this.node.uniqueId}:Permissions`, {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.authorizerArn,
            });
        }
        else if (this.role instanceof iam.Role) { // i.e. not imported
            this.role.attachInlinePolicy(new iam.Policy(this, 'authorizerInvokePolicy', {
                statements: [
                    new iam.PolicyStatement({
                        resources: [this.handler.functionArn],
                        actions: ['lambda:InvokeFunction'],
                    }),
                ],
            }));
        }
    }
    /**
     * Returns a token that resolves to the Rest Api Id at the time of synthesis.
     * Throws an error, during token resolution, if no RestApi is attached to this authorizer.
     */
    lazyRestApiId() {
        return core_1.Lazy.stringValue({
            produce: () => {
                if (!this.restApiId) {
                    throw new Error(`Authorizer (${this.node.path}) must be attached to a RestApi`);
                }
                return this.restApiId;
            },
        });
    }
}
/**
 * Token based lambda authorizer that recognizes the caller's identity as a bearer token,
 * such as a JSON Web Token (JWT) or an OAuth token.
 * Based on the token, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class TokenAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'TOKEN',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySource || 'method.request.header.Authorization',
            identityValidationExpression: props.validationRegex,
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.TokenAuthorizer = TokenAuthorizer;
/**
 * Request-based lambda authorizer that recognizes the caller's identity via request parameters,
 * such as headers, paths, query strings, stage variables, or context variables.
 * Based on the request, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class RequestAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        if ((props.resultsCacheTtl === undefined || props.resultsCacheTtl.toSeconds() !== 0) && props.identitySources.length === 0) {
            throw new Error('At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        }
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'REQUEST',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySources.map(is => is.toString()).join(','),
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.RequestAuthorizer = RequestAuthorizer;
/**
 * constructs the authorizerURIArn.
 */
function lambdaAuthorizerArn(handler) {
    return `arn:${core_1.Stack.of(handler).partition}:apigateway:${core_1.Stack.of(handler).region}:lambda:path/2015-03-31/functions/${handler.functionArn}/invocations`;
}
//# sourceMappingURL=data:application/json;base64,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