"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Represents an identity source.
 *
 * The source can be specified either as a literal value (e.g: `Auth`) which
 * cannot be blank, or as an unresolved string token.
 */
class IdentitySource {
    /**
     * Provides a properly formatted header identity source.
     * @param headerName the name of the header the `IdentitySource` will represent.
     *
     * @returns a header identity source.
     */
    static header(headerName) {
        return IdentitySource.toString(headerName, 'method.request.header');
    }
    /**
     * Provides a properly formatted query string identity source.
     * @param queryString the name of the query string the `IdentitySource` will represent.
     *
     * @returns a query string identity source.
     */
    static queryString(queryString) {
        return IdentitySource.toString(queryString, 'method.request.querystring');
    }
    /**
     * Provides a properly formatted API Gateway stage variable identity source.
     * @param stageVariable the name of the stage variable the `IdentitySource` will represent.
     *
     * @returns an API Gateway stage variable identity source.
     */
    static stageVariable(stageVariable) {
        return IdentitySource.toString(stageVariable, 'stageVariables');
    }
    /**
     * Provides a properly formatted request context identity source.
     * @param context the name of the context variable the `IdentitySource` will represent.
     *
     * @returns a request context identity source.
     */
    static context(context) {
        return IdentitySource.toString(context, 'context');
    }
    static toString(source, type) {
        if (!source.trim()) {
            throw new Error('IdentitySources must be a non-empty string.');
        }
        return `${type}.${source}`;
    }
}
exports.IdentitySource = IdentitySource;
//# sourceMappingURL=data:application/json;base64,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