"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
/**
 * Basic Auth configuration
 */
class BasicAuth {
    constructor(props) {
        this.props = props;
    }
    /**
     * Creates a Basic Auth configuration from a username and a password
     *
     * @param username The username
     * @param password The password
     */
    static fromCredentials(username, password) {
        return new BasicAuth({ username, password });
    }
    /**
     * Creates a Basic Auth configuration with a password generated in Secrets
     * Manager.
     *
     * @param username The username
     * @param encryptionKey The encryption key to use to encrypt the password in
     * Secrets Manager
     */
    static fromGeneratedPassword(username, encryptionKey) {
        return new BasicAuth({ username, encryptionKey });
    }
    /**
     * Binds this Basic Auth configuration to an App
     */
    bind(scope, id) {
        const config = {
            enableBasicAuth: true,
            username: this.props.username,
        };
        if (this.props.password) {
            return {
                ...config,
                password: this.props.password.toString(),
            };
        }
        const secret = new secretsmanager.Secret(scope, id, {
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: this.props.username }),
                generateStringKey: 'password',
            },
        });
        return {
            ...config,
            password: secret.secretValueFromJson('password').toString(),
        };
    }
}
exports.BasicAuth = BasicAuth;
//# sourceMappingURL=data:application/json;base64,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