import unittest
from parameterized import parameterized
from iqrfpy.enums.Commands import CoordinatorResponseCommands
from iqrfpy.enums.MessageTypes import CoordinatorMessages
from iqrfpy.enums.peripherals import EmbedPeripherals
from iqrfpy.messages.responses.coordinator.SetDpaParams import SetDpaParamsResponse, DpaParam

data_ok: dict = {
    'msgid': 'setDpaParamsTest',
    'nadr': 0,
    'hwpid': 0,
    'rcode': 0,
    'dpa_value': 64,
    'dpa_param': DpaParam.LAST_RSSI,
    'dpa': b'\x00\x00\x00\x88\x00\x00\x00\x40\x00'
}

data_ok_1: dict = {
    'msgid': 'setDpaParamsTest',
    'nadr': 0,
    'hwpid': 1026,
    'rcode': 0,
    'dpa_value': 35,
    'dpa_param': DpaParam.USER_SPECIFIED,
    'dpa': b'\x00\x00\x00\x88\x02\x04\x00\x23\x03'
}

data_error: dict = {
    'msgid': 'setDpaParamsTest',
    'nadr': 0,
    'hwpid': 1028,
    'rcode': 1,
    'dpa_value': 35,
    'dpa': b'\x00\x00\x00\x88\x04\x04\x01\x23'
}


class SetDpaParamsResponseTestCase(unittest.TestCase):

    @staticmethod
    def generate_json(response_data: dict):
        json = {
            'mType': 'iqrfEmbedCoordinator_AddrInfo',
            'data': {
                'msgId': response_data['msgid'],
                'rsp': {
                    'nAdr': response_data['nadr'],
                    'hwpId': response_data['hwpid'],
                    'rCode': response_data['rcode'],
                    'dpaVal': response_data['dpa_value'],
                },
                'insId': 'iqrfgd2-1',
                'status': response_data['rcode']
            }
        }
        if response_data['rcode'] == 0:
            json['data']['rsp']['result'] = {
                'prevDpaParam': response_data['dpa_param']
            }
        return json

    @parameterized.expand([
        ['from_dpa', data_ok, SetDpaParamsResponse.from_dpa(data_ok['dpa']), False],
        ['from_dpa', data_ok_1, SetDpaParamsResponse.from_dpa(data_ok_1['dpa']), False],
        ['from_json', data_ok, SetDpaParamsResponse.from_json(generate_json(data_ok)), True],
        ['from_json', data_ok_1, SetDpaParamsResponse.from_json(generate_json(data_ok_1)), True],
        ['from_dpa_error', data_error, SetDpaParamsResponse.from_dpa(data_error['dpa']), False],
        ['from_json_error', data_error, SetDpaParamsResponse.from_json(generate_json(data_error)), True]
    ])
    def test_factory_methods_ok(self, _, response_data, response, json):
        with self.subTest():
            self.assertEqual(response.get_nadr(), response_data['nadr'])
        with self.subTest():
            self.assertEqual(response.get_pnum(), EmbedPeripherals.COORDINATOR)
        with self.subTest():
            self.assertEqual(response.get_pcmd(), CoordinatorResponseCommands.SET_DPA_PARAMS)
        with self.subTest():
            self.assertEqual(response.get_hwpid(), response_data['hwpid'])
        with self.subTest():
            self.assertEqual(response.get_rcode(), response_data['rcode'])
        if json:
            with self.subTest():
                self.assertEqual(response.get_mtype(), CoordinatorMessages.SET_DPA_PARAMS)
            with self.subTest():
                self.assertEqual(response.get_msgid(), response_data['msgid'])

    def test_from_dpa_invalid(self):
        with self.assertRaises(ValueError):
            SetDpaParamsResponse.from_dpa(b'\x00\x00\x00\x88\x00\x00\x00\x40')

    @parameterized.expand([
        ['from_dpa', data_ok['dpa_param'], SetDpaParamsResponse.from_dpa(data_ok['dpa'])],
        ['from_dpa', data_ok_1['dpa_param'], SetDpaParamsResponse.from_dpa(data_ok_1['dpa'])],
        ['from_json', data_ok['dpa_param'], SetDpaParamsResponse.from_json(generate_json(data_ok))],
        ['from_json', data_ok_1['dpa_param'], SetDpaParamsResponse.from_json(generate_json(data_ok_1))]
    ])
    def test_get_dpa_param(self, _, dpa_param: DpaParam, response):
        self.assertEqual(response.get_dpa_param(), dpa_param)
