"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.shutdownTerminal = exports.listRunning = exports.startNew = exports.isAvailable = exports.TERMINAL_SERVICE_URL = void 0;
const coreutils_1 = require("@jupyterlab/coreutils");
const serverconnection_1 = require("../serverconnection");
/**
 * The url for the terminal service.
 */
exports.TERMINAL_SERVICE_URL = 'api/terminals';
/**
 * Whether the terminal service is available.
 */
function isAvailable() {
    const available = String(coreutils_1.PageConfig.getOption('terminalsAvailable'));
    return available.toLowerCase() === 'true';
}
exports.isAvailable = isAvailable;
/**
 * Start a new terminal session.
 *
 * @param settings - The server settings to use.
 *
 * @param name - The name of the target terminal.
 *
 * @param cwd - The path in which the terminal will start.
 *
 * @returns A promise that resolves with the session model.
 */
async function startNew(settings = serverconnection_1.ServerConnection.makeSettings(), name, cwd) {
    Private.errorIfNotAvailable();
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.TERMINAL_SERVICE_URL);
    const init = {
        method: 'POST',
        body: JSON.stringify({ name, cwd })
    };
    const response = await serverconnection_1.ServerConnection.makeRequest(url, init, settings);
    if (response.status !== 200) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
    const data = await response.json();
    // TODO: Validate model
    return data;
}
exports.startNew = startNew;
/**
 * List the running terminal sessions.
 *
 * @param settings - The server settings to use.
 *
 * @returns A promise that resolves with the list of running session models.
 */
async function listRunning(settings = serverconnection_1.ServerConnection.makeSettings()) {
    Private.errorIfNotAvailable();
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.TERMINAL_SERVICE_URL);
    const response = await serverconnection_1.ServerConnection.makeRequest(url, {}, settings);
    if (response.status !== 200) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
    const data = await response.json();
    if (!Array.isArray(data)) {
        throw new Error('Invalid terminal list');
    }
    // TODO: validate each model
    return data;
}
exports.listRunning = listRunning;
/**
 * Shut down a terminal session by name.
 *
 * @param name - The name of the target session.
 *
 * @param settings - The server settings to use.
 *
 * @returns A promise that resolves when the session is shut down.
 */
async function shutdownTerminal(name, settings = serverconnection_1.ServerConnection.makeSettings()) {
    var _a;
    Private.errorIfNotAvailable();
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.TERMINAL_SERVICE_URL, name);
    const init = { method: 'DELETE' };
    const response = await serverconnection_1.ServerConnection.makeRequest(url, init, settings);
    if (response.status === 404) {
        const data = await response.json();
        const msg = (_a = data.message) !== null && _a !== void 0 ? _a : `The terminal session "${name}"" does not exist on the server`;
        console.warn(msg);
    }
    else if (response.status !== 204) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
}
exports.shutdownTerminal = shutdownTerminal;
var Private;
(function (Private) {
    /**
     * Throw an error if terminals are not available.
     */
    function errorIfNotAvailable() {
        if (!isAvailable()) {
            throw new Error('Terminals Unavailable');
        }
    }
    Private.errorIfNotAvailable = errorIfNotAvailable;
})(Private || (Private = {}));
//# sourceMappingURL=restapi.js.map