"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const codebuild_1 = require("./providers/codebuild");
const fargate_1 = require("./providers/fargate");
const lambda_1 = require("./providers/lambda");
const secrets_1 = require("./secrets");
const utils_1 = require("./utils");
const webhook_1 = require("./webhook");
/**
 * Create all the required infrastructure to provide self-hosted GitHub runners. It creates a webhook, secrets, and a step function to orchestrate all runs. Secrets are not automatically filled. See README.md for instructions on how to setup GitHub integration.
 *
 * By default, this will create a runner provider of each available type with the defaults. This is good enough for the initial setup stage when you just want to get GitHub integration working.
 *
 * ```typescript
 * new GitHubRunners(stack, 'runners', {});
 * ```
 *
 * Usually you'd want to configure the runner providers so the runners can run in a certain VPC or have certain permissions.
 *
 * ```typescript
 * const vpc = ec2.Vpc.fromLookup(stack, 'vpc', { vpcId: 'vpc-1234567' });
 * const runnerSg = new ec2.SecurityGroup(stack, 'runner security group', { vpc: vpc });
 * const dbSg = ec2.SecurityGroup.fromSecurityGroupId(stack, 'database security group', 'sg-1234567');
 * const bucket = new s3.Bucket(stack, 'runner bucket');
 *
 * // create a custom CodeBuild provider
 * const myProvider = new CodeBuildRunner(
 *   stack, 'codebuild runner',
 *   {
 *      label: 'my-codebuild',
 *      vpc: vpc,
 *      securityGroup: runnerSg,
 *   },
 * );
 * // grant some permissions to the provider
 * bucket.grantReadWrite(myProvider);
 * dbSg.connections.allowFrom(runnerSg, ec2.Port.tcp(3306), 'allow runners to connect to MySQL database');
 *
 * // create the runner infrastructure
 * new GitHubRunners(
 *   stack,
 *   'runners',
 *   {
 *     providers: [myProvider],
 *     defaultProviderLabel: 'my-codebuild',
 *   }
 * );
 * ```
 */
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        if (this.props.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new codebuild_1.CodeBuildRunner(this, 'CodeBuild', {}),
                new lambda_1.LambdaRunner(this, 'Lambda', {}),
                new fargate_1.FargateRunner(this, 'Fargate', {}),
            ];
        }
        const defaultProvider = this.getDefaultProvider();
        if (!defaultProvider) {
            throw new Error(`No provider was found for the default label "${this.props.defaultProviderLabel}"`);
        }
        else {
            this.defaultProvider = defaultProvider;
        }
        this.orchestrator = this.stateMachine();
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
        });
        this.statusFunctions();
    }
    getDefaultProvider() {
        for (const provider of this.providers) {
            if ((this.props.defaultProviderLabel || 'codebuild') == provider.label) {
                return provider;
            }
        }
        return null;
    }
    stateMachine() {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        const deleteRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Runner', {
            lambdaFunction: this.deleteRunner(),
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
            }),
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.isPresent(`$.labels.${provider.label}`), providerTask);
            if (this.defaultProvider == provider) {
                providerChooser.otherwise(providerTask);
            }
        }
        const work = tokenRetrieverTask.next(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: '$.result' })
            .branch(providerChooser)
            .addCatch(deleteRunnerTask
            .next(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Runner Failed')), {
            resultPath: '$.error',
        }));
        const check = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Is self hosted?')
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.isNotPresent('$.labels.self-hosted'), new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'No'))
            .otherwise(work);
        return new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definition: check,
        });
    }
    tokenRetriever() {
        const func = new utils_1.BundledNodejsFunction(this, 'token-retriever', {
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
            },
            timeout: cdk.Duration.seconds(30),
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteRunner() {
        const func = new utils_1.BundledNodejsFunction(this, 'delete-runner', {
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
            },
            timeout: cdk.Duration.seconds(30),
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunctions() {
        const func = this.providers.map(provider => {
            return {
                type: provider.constructor.name,
                label: provider.label,
                vpcArn: provider.vpc && provider.vpc.vpcArn,
                securityGroup: provider.securityGroup && provider.securityGroup.securityGroupId,
                roleArn: provider.grantPrincipal.grantPrincipal.roleArn,
            };
        });
        const statusFunction = new utils_1.BundledNodejsFunction(this, 'status', {
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
                PROVIDERS: JSON.stringify(func),
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
            },
            timeout: cdk.Duration.minutes(3),
        });
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${cdk.Stack.of(this).region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
    }
}
exports.GitHubRunners = GitHubRunners;
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.0.13" };
//# sourceMappingURL=data:application/json;base64,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