"""
    This is a script that illustrates grid search cross validation to optimize parameters for a 2D U-Net
"""

"""
    Necessary libraries
"""
import argparse
import yaml

from neuralnets.data.datasets import LabeledVolumeDataset, LabeledSlidingWindowDataset
from neuralnets.util.augmentation import *
from neuralnets.util.io import print_frm, save
from neuralnets.util.tools import set_seed, parse_params
from neuralnets.util.losses import get_loss_function
from neuralnets.cross_validation.base import UNet2DClassifier, parse_search_grid

from multiprocessing import freeze_support
from sklearn.model_selection import GridSearchCV

if __name__ == '__main__':
    freeze_support()

    """
        Parse all the arguments
    """
    print_frm('Parsing arguments')
    parser = argparse.ArgumentParser()
    parser.add_argument("--config", "-c", help="Path to the configuration file", type=str, default='unet_2d.yaml')
    args = parser.parse_args()
    with open(args.config) as file:
        params = parse_params(yaml.load(file, Loader=yaml.FullLoader))

    """
    Fix seed (for reproducibility)
    """
    set_seed(params['seed'])

    """
        Load the data
    """
    print_frm('Loading data')
    input_shape = (1, *(params['input_size']))
    transform = Compose([Rotate90(), Flip(prob=0.5, dim=0), Flip(prob=0.5, dim=1), ContrastAdjust(adj=0.1),
                         RandomDeformation(), AddNoise(sigma_max=0.05), CleanDeformedLabels(params['coi'])])
    train = LabeledVolumeDataset(params['train_data'], params['train_labels'], input_shape=input_shape,
                                 type=params['data_type'], batch_size=params['train_batch_size'], transform=transform)
    val = LabeledSlidingWindowDataset(params['val_data'], params['val_labels'], input_shape=input_shape,
                                      type=params['data_type'], batch_size=params['train_batch_size'])
    X_train = np.concatenate((train.data, val.data), axis=0)
    y_train = np.concatenate((train.labels, val.labels), axis=0)

    """
        Build the network
    """
    print_frm('Building the network')
    loss_fn = get_loss_function(params['loss'])
    clf = UNet2DClassifier(epochs=params['epochs'], gpus=params['gpus'], accelerator=params['accelerator'],
                           log_dir=params['log_dir'], log_freq=params['log_freq'],
                           log_refresh_rate=params['log_refresh_rate'], train_batch_size=params['train_batch_size'],
                           test_batch_size=params['test_batch_size'], num_workers=params['num_workers'],
                           device=params['gpus'][0], transform=transform, feature_maps=params['fm'],
                           levels=params['levels'], dropout_enc=params['dropout'], dropout_dec=params['dropout'],
                           norm=params['norm'], activation=params['activation'], coi=params['coi'], loss_fn=loss_fn)

    """
        Perform cross validation grid search
    """
    print_frm('Starting grid search cross validation')
    search_grid = parse_search_grid(params['search_grid'])
    gs = GridSearchCV(clf, search_grid, cv=params['folds'])
    gs.fit(X_train, y_train)

    """
        Save and report results
    """
    save(gs.cv_results_, params['results_file'])
    print_frm(gs.best_params_)
    print_frm('Best mIoU: %.6f' % gs.best_score_)

    print_frm('Finished!')
