#!/usr/bin/env python3
import signal
import time
import sys
from argparse import ArgumentParser
from prometheus_client import start_http_server, Gauge
import habets39  # pytype: disable=import-error
from gnuradio import eng_notation  # pytype: disable=import-error
from gnuradio import gr  # pytype: disable=import-error
from gnuradio.eng_arg import eng_float  # pytype: disable=import-error
from gnuradio.eng_arg import intx  # pytype: disable=import-error
from gamutrf.grscan import grscan

from gamutrf.utils import ETTUS_ARGS


def init_prom_vars():
    prom_vars = {
        'freq_start_hz': Gauge('freq_start_hz', 'start of scanning range in Hz'),
        'freq_end_hz': Gauge('freq_end_hz', 'end of scanning range in Hz'),
        'sweep_sec': Gauge('sweep_sec', 'scan sweep rate in seconds'),
        'last_freq_update': Gauge('last_freq_update', 'last frequency update time'),
    }
    return prom_vars


def argument_parser():
    parser = ArgumentParser()
    parser.add_argument(
        '--freq-end', dest='freq_end', type=eng_float, default=eng_notation.num_to_str(float(1e9)),
        help='Set freq_end [default=%(default)r]')
    parser.add_argument(
        '--freq-start', dest='freq_start', type=eng_float, default=eng_notation.num_to_str(float(100e6)),
        help='Set freq_start [default=%(default)r]')
    parser.add_argument(
        '--igain', dest='igain', type=intx, default=0,
        help='Set igain[default=%(default)r]')
    parser.add_argument(
        '--samp-rate', dest='samp_rate', type=eng_float, default=eng_notation.num_to_str(float(4e6)),
        help='Set samp_rate [default=%(default)r]')
    parser.add_argument(
        '--sweep-sec', dest='sweep_sec', type=intx, default=30,
        help='Set sweep_sec [default=%(default)r]')
    parser.add_argument(
        '--logaddr', dest='logaddr', type=str, default='127.0.0.1',
        help='Log UDP results to this address')
    parser.add_argument(
        '--logport', dest='logport', type=int, default=8001,
        help='Log UDP results to this port')
    parser.add_argument(
        '--promport', dest='promport', type=int, default=9000,
        help='Prometheus client port')
    parser.add_argument(
        '--sdr', dest='sdr', type=str, default='ettus',
        help='SDR to use (ettus, bladerf, or lime)')
    parser.add_argument(
        '--ettusargs', dest='ettusargs', type=str, default=ETTUS_ARGS,
        help='extra args to pass to Ettus driver')
    parser.add_argument(
        '--updatetimeout', dest='updatetimeout', type=int, default=10,
        help='seconds to wait for healthy freq updates')
    return parser


def main():
    options = argument_parser().parse_args()
    if gr.enable_realtime_scheduling() != gr.RT_OK:
        print('Warning: failed to enable real-time scheduling.')

    if options.freq_start > options.freq_end:
        print('Error: freq_start must be less than freq_end')
        sys.exit(1)

    if options.freq_end > 6e9:
        print('Error: freq_end must be less than 6GHz')
        sys.exit(1)

    prom_vars = init_prom_vars()
    prom_vars['freq_start_hz'].set(options.freq_start)
    prom_vars['freq_end_hz'].set(options.freq_end)
    prom_vars['sweep_sec'].set(options.sweep_sec)
    start_http_server(options.promport)

    tb = grscan(freq_end=options.freq_end, freq_start=options.freq_start,
                igain=options.igain, samp_rate=options.samp_rate,
                sweep_sec=options.sweep_sec,
                logaddr=options.logaddr, logport=options.logport,
                sdr=options.sdr, ettusargs=options.ettusargs,
                habets39=habets39)

    def sig_handler(_sig=None, _frame=None):
        tb.stop()
        tb.wait()
        sys.exit(0)

    signal.signal(signal.SIGINT, sig_handler)
    signal.signal(signal.SIGTERM, sig_handler)

    tb.start()
    while True:
        prom_vars['last_freq_update'].set(tb.freq_update)
        time.sleep(1)
        if not tb.freq_updated(options.updatetimeout):
            tb.stop()
            break

    tb.wait()


if __name__ == '__main__':
    main()
