"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationDeployStackSetAction = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const action_1 = require("../action");
const common_1 = require("../common");
const singleton_policy_1 = require("./private/singleton-policy");
const stackset_types_1 = require("./stackset-types");
/**
 * CodePipeline action to deploy a stackset.
 *
 * CodePipeline offers the ability to perform AWS CloudFormation StackSets
 * operations as part of your CI/CD process. You use a stack set to create
 * stacks in AWS accounts across AWS Regions by using a single AWS
 * CloudFormation template. All the resources included in each stack are defined
 * by the stack set’s AWS CloudFormation template. When you create the stack
 * set, you specify the template to use, as well as any parameters and
 * capabilities that the template requires.
 *
 * For more information about concepts for AWS CloudFormation StackSets, see
 * [StackSets
 * concepts](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html)
 * in the AWS CloudFormation User Guide.
 *
 * If you use this action to make an update that includes adding stack
 * instances, the new instances are deployed first and the update is completed
 * last. The new instances first receive the old version, and then the update is
 * applied to all instances.
 *
 * As a best practice, you should construct your pipeline so that the stack set
 * is created and initially deploys to a subset or a single instance. After you
 * test your deployment and view the generated stack set, then add the
 * CloudFormationStackInstances action so that the remaining instances are
 * created and updated.
 */
class CloudFormationDeployStackSetAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            region: props.stackSetRegion,
            provider: 'CloudFormationStackSet',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 1,
                maxInputs: 3,
                minOutputs: 0,
                maxOutputs: 0,
            },
            inputs: [
                ...props.template._artifactsReferenced ?? [],
                ...props.parameters?._artifactsReferenced ?? [],
                ...props.stackInstances?._artifactsReferenced ?? [],
            ],
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_actions_CloudFormationDeployStackSetActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudFormationDeployStackSetAction);
            }
            throw error;
        }
        this.props = props;
        this.deploymentModel = props.deploymentModel ?? stackset_types_1.StackSetDeploymentModel.selfManaged();
        common_1.validatePercentage('failureTolerancePercentage', props.failureTolerancePercentage);
        common_1.validatePercentage('maxAccountConcurrencyPercentage', props.maxAccountConcurrencyPercentage);
    }
    bound(scope, _stage, options) {
        const singletonPolicy = singleton_policy_1.SingletonPolicy.forRole(options.role);
        singletonPolicy.grantCreateUpdateStackSet(this.props);
        const instancesResult = this.props.stackInstances?._bind(scope);
        const permissionModelBind = this.deploymentModel?._bind(scope);
        for (const role of permissionModelBind?.passedRoles ?? []) {
            singletonPolicy.grantPassRole(role);
        }
        if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackSetName: this.props.stackSetName,
                Description: this.props.description,
                TemplatePath: this.props.template._render(),
                Parameters: this.props.parameters?._render(),
                Capabilities: singleton_policy_1.parseCapabilities(this.props.cfnCapabilities),
                FailureTolerancePercentage: this.props.failureTolerancePercentage,
                MaxConcurrentPercentage: this.props.maxAccountConcurrencyPercentage,
                ...instancesResult?.stackSetConfiguration,
                ...permissionModelBind?.stackSetConfiguration,
            },
        };
    }
}
exports.CloudFormationDeployStackSetAction = CloudFormationDeployStackSetAction;
_a = JSII_RTTI_SYMBOL_1;
CloudFormationDeployStackSetAction[_a] = { fqn: "@aws-cdk/aws-codepipeline-actions.CloudFormationDeployStackSetAction", version: "1.163.1" };
//# sourceMappingURL=data:application/json;base64,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